package com.st.stellar.component.model.editor

import java.util.Collection
import org.eclipse.emf.common.notify.Notifier
import org.eclipse.emf.common.util.Diagnostic
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.emfforms.spi.swt.core.ui.SWTValidationHelper
import org.eclipse.emfforms.spi.swt.treemasterdetail.diagnostic.DiagnosticCache
import org.eclipse.jface.resource.DeviceResourceManager
import org.eclipse.jface.resource.ImageDescriptor
import org.eclipse.jface.resource.ResourceManager
import org.eclipse.jface.viewers.DecorationOverlayIcon
import org.eclipse.jface.viewers.ILabelDecorator
import org.eclipse.jface.viewers.ILabelProviderListener
import org.eclipse.jface.viewers.TreeViewer
import org.eclipse.swt.graphics.Image
import org.eclipse.swt.graphics.Point
import org.eclipse.swt.graphics.Rectangle
import com.st.stellar.component.Application

class ComponentLabelServiceDecorator implements ILabelDecorator {
	final MyDiagnosticCache cache
	final TreeViewer viewer
	final ResourceManager images

	/** 
	 * Default constructor.
	 * @param viewer the {@link TreeViewer}
	 * @param input the input notifier
	 * @param cache the {@link DiagnosticCache}
	 */
	new(TreeViewer viewer, Notifier input, MyDiagnosticCache cache) {
		this.viewer = viewer
		this.cache = cache
		images = new DeviceResourceManager(viewer.getControl().getDisplay())
		cache.registerValidationListener([ Collection<EObject> object, boolean potentialStructuralChange |
			if (potentialStructuralChange) {
				for (EObject o : object) {
					refreshViewer(o)
				}
			} else {
				for (EObject o : object) {
					updateViewer(o)
				}
				if( !object.isEmpty ) {
					val comp = EcoreUtil.getRootContainer(object.get(0)) as EObject;
					cache.updateCache(comp);
				}
			}
		])
		viewer.refresh()
	}

	override Image decorateImage(Image image, Object element) {
		if (image === null) {
			return image
		}
		if (!EObject.isInstance(element) && !Resource.isInstance(element)) {
			return image
		}
		val Diagnostic diagnostic = cache.getCachedValue(element)
		val int severity = diagnostic.getSeverity()
		val ImageDescriptor validationOverlayDescriptor = SWTValidationHelper.INSTANCE.
			getValidationOverlayDescriptor(severity)
		if (validationOverlayDescriptor === null) {
			return image
		}
		val Rectangle bounds = image.getBounds()
		val Point size = new Point(bounds.width, bounds.height)
		val DecorationOverlayIcon icon = new DecorationOverlayIcon(image, #[validationOverlayDescriptor], size)
		return (images.get(icon) as Image)
	}

	/** 
	 * Called in order to update the cache. This also triggers a viewer refresh.
	 * @param element The element which changed
	 */
	def protected void refreshViewer(EObject element) {
		viewer.refresh(element, true)
	}

	/** 
	 * Called in order to update the cache. This also triggers a viewer update.
	 * @param element The element which changed
	 */
	def protected void updateViewer(EObject element) {
		viewer.update(element, null)
	}

	override String decorateText(String text, Object element) {
		return element.format(text)
	}

	override void addListener(ILabelProviderListener listener) {
		/* no op */
	}

	override boolean isLabelProperty(Object element, String property) {
		/* no op */
		return false
	}

	override void removeListener(ILabelProviderListener listener) {
		/* no op */
	}

	override void dispose() {
		cache.dispose()
		images.dispose()
	}

	dispatch def format(Application object, String text) {
		return object.name
	}

	dispatch def format(EObject object, String text) {
		return text
	}

	dispatch def format(Object object, String text) {
		return text
	}

}
