package com.st.stellar.component.API

import com.google.common.io.Files
import com.st.stellar.component.Application
import java.io.File
import java.net.URLDecoder
import java.nio.charset.Charset
import java.util.ArrayList
import java.util.HashSet
import java.util.List
import java.util.Set
import org.apache.log4j.Logger
import org.eclipse.core.resources.IContainer
import org.eclipse.core.resources.IFile
import org.eclipse.core.resources.IFolder
import org.eclipse.core.resources.IProject
import org.eclipse.core.resources.IResource
import org.eclipse.core.resources.IResourceProxy
import org.eclipse.core.resources.IResourceProxyVisitor
import org.eclipse.core.resources.IResourceVisitor
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.CoreException
import org.eclipse.core.runtime.IPath
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.core.runtime.Path
import org.eclipse.emf.codegen.ecore.CodeGenEcorePlugin
import org.eclipse.emf.common.util.Diagnostic
import org.eclipse.emf.common.util.Monitor
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.util.EcoreUtil

class Utils {

	static final Logger LOG = Logger.getLogger(Utils);

	static def IFile getFileFromResource(URI uri) {
		// Convert the URI to a platform resource URI
		if (uri.isPlatformResource()) {
			val platformString = uri.toPlatformString(true)

			// Step 3: Get the workspace root
			val workspaceRoot = ResourcesPlugin.workspace.root

			// Step 4: Find the file in the workspace
			val file = workspaceRoot.getFile(new Path(platformString))
			return file
		}
		if (uri.isFile()) {
			val workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
			val files = workspaceRoot.findFilesForLocationURI(new java.net.URI(uri.toString));
			if (files.length > 0) {
				return files.get(0); // Return the first matching file
			}
		}

		// If the URI is not a platform resource URI, return null or handle accordingly
		return null
	}

	private static def URI getResolvedImportUri(Resource context, URI uri) {
		var uriRes = uri
		val contextURI = context.getURI();
		if (contextURI.isHierarchical() && !contextURI.isRelative() && (uri.isRelative() && !uri.isEmpty())) {
			uriRes = uri.resolve(contextURI);
		}
		return uriRes;
	}

	static def Resource getResource(Resource context, String uriStr) {
		var Resource res = null
		if (uriStr !== null && context !== null) {
			try {
				val newURI = getResolvedImportUri(context, URI.createURI(uriStr));
				val rs = context.resourceSet; // new ResourceSetImpl();
				val r = rs.getResource(newURI, true);
				if (r !== null) {
					r.load(context.resourceSet.loadOptions);
					EcoreUtil.resolveAll(r)
				}
				if (r === null) {
					return null;
				}
				if (r.getContents().size() > 0) {
					res = r;
				}

			} catch (Exception e) {
				LOG.error(e.getMessage());
			}
		}
		return res;
	}

	static class ProjectVisitor implements IResourceProxyVisitor {

		List<IPath> _list;
		String _extensions;

		new(List<IPath> list, String ext) {
			_list = list;
			_extensions = ext
		}

		override visit(IResourceProxy proxy) throws CoreException {
			if (proxy.getType() == IResource.FILE) {
				val path = new Path(proxy.getName())
				if (checkComponent(path)) {
					_list.add(path)
				}
//			} else {
//				System.out.println(proxy.getType())
			}
			return true;
		}

		def boolean checkComponent(Path path) {
			for (e : _extensions.split(",")) {
				if (path.toString.endsWith(e)) {
					return true;
				}
			}
		}
	}

	static def List<Application> getApplicationModels(Resource context, IProject project, String suffix) {
		val files = new ArrayList<IPath>()

		val workspaceRoot = ResourcesPlugin.getWorkspace().getRoot()
		val resourceSet = context.resourceSet // new ResourceSetImpl()
		val visitor = new ProjectVisitor(files, suffix)
		val handles = new ArrayList<Application>()
		try {
			project.accept(visitor, IResource.NONE);
		} catch (CoreException e) {
			LOG.error(e.getMessage());
			return handles;
		}
		val iter = files.iterator();
		while (iter.hasNext()) {
			val p = iter.next() as IPath
			val path = new Path("/" + project.getName() + "/" + p.toString());
			val f = workspaceRoot.getFile(path);

			var String decodedUri = ""
			try {
				// System.out.println(f.getProjectRelativePath().toOSString());
				decodedUri = URLDecoder.decode(f.getProjectRelativePath().toOSString(), "UTF-8");

				val loadResource = getResource(context, decodedUri);
				if (loadResource !== null) {
					loadResource.load(resourceSet.getLoadOptions());
					val contents = loadResource.getContents();
					if (!contents.isEmpty) {
						val comp = contents.get(0);
						if (Application.isInstance(comp)) {
							handles.add(Application.cast(comp));
						}
					}
				}
			} catch (IllegalStateException e) {
				// TODO Auto-generated catch block
				LOG.error(e.message)
			}
		}
		return handles;
	}

	static def List<IFile> findFilesWithExtension(IProject project, String ext) {
		val files = newArrayList
		try {
			project.accept(new IResourceVisitor() {
				override boolean visit(IResource resource) throws CoreException {
					if (resource !== null && resource instanceof IFile && resource.getFileExtension() !== null &&
						resource.getFileExtension().equals(ext)) {
						files.add(resource as IFile)
					}
					true
				}
			})
		} catch (CoreException e) {
			// e.printStackTrace()
		}
		files
	}

	static def String populateDiagnostics(Diagnostic diag) {
		return populateDiagnostics(diag, -1)
	}
	static def String populateDiagnostics(Diagnostic diag, int filterCode) {
		val diags = new HashSet
		populateDiagnostics(diag, diags, filterCode)
		val mess = new StringBuffer();
		diags.filter[filterCode == -1 || code == filterCode].map[message].forEach(it|mess.append(it).append("\n"));
		mess.toString
	}

	static private def void populateDiagnostics(Diagnostic diag, Set<Diagnostic> errorsList, int filterCode) {
		val errors = diag.children
		if (errors !== null) {
			if (errors.size > 0) {
				for (Diagnostic e : errors) {
					errorsList.add(e)
					populateDiagnostics(e, errorsList, filterCode)
				}
			} else {
				errorsList.add(diag)
			}
		}
	}

	static def generateText(EObject eobject, String targetDir, String fileName, CharSequence text, Monitor monitor) {
		try {
			monitor.beginTask("", 3)

			try {
				// Get project from EObject
				val IProject project = getProjectFromEObject(eobject)
				val IFolder folder = project.getFolder(targetDir)
				if (!folder.exists) {
					createRecursive(folder)
				}
				val IFile ifile = project.getFile(targetDir + "/" + fileName)
				val File file = ifile.rawLocation.makeAbsolute.toFile
				if (!file.exists) {
					file.createNewFile
				}
				println("save to: " + file.toString)
				Files.asCharSink(file, Charset.defaultCharset).write(text)
			} catch (Exception exception) {
				CodeGenEcorePlugin.INSTANCE.log(exception)
			}
		} finally {
			monitor.done
		}
	}

	static def void createRecursive(IContainer resource) {
        if (!resource.exists()) {
            if (!resource.getParent().exists()) {
                createRecursive(resource.getParent());
            }
            if (resource instanceof IFolder) {
                try {
                    IFolder.cast(resource).create(false, true, new NullProgressMonitor());
                } catch (CoreException e) {
                    CodeGenEcorePlugin.INSTANCE.log(e);
                }
            }
        }
    }
    	
	def static IProject getProjectFromEObject(EObject eObject) {
		if (eObject === null) {
			return null
		}

		val Resource resource = eObject.eResource
		if (resource === null) {
			return null
		}

		val URI uri = resource.URI
		if (uri === null || !uri.isPlatformResource) {
			return null
		}

		val String platformString = uri.toPlatformString(true)
		val IFile file = ResourcesPlugin.workspace.root.getFile(new Path(platformString))
		if (file === null) {
			return null
		}

		return file.project
	}


	static def removeDuplicates(List<String> stringList) {
		// Using LinkedHashSet to preserve order and remove duplicates
		val uniqueSet = newLinkedHashSet(stringList).get(0).toSet
		val uniqueList = uniqueSet.toList
		uniqueList
	}

	static boolean debugOption = false
	static def void setDebugOption(boolean debug) {
		debugOption = debug
	}
	static def boolean getDebugOption() {
		debugOption
	}


}
