package com.st.stellar.component.API

import com.st.stellar.component.Application
import com.st.stellar.component.Component
import com.st.stellar.component.ComponentPackage
import com.st.stellar.component.ComponentReference
import com.st.stellar.component.DerivedAttributesNotifier
import java.lang.reflect.InvocationTargetException
import java.net.URLDecoder
import java.util.ArrayList
import java.util.Collection
import java.util.Collections
import java.util.HashMap
import java.util.HashSet
import java.util.List
import java.util.Map
import java.util.Set
import org.apache.log4j.Level
import org.apache.log4j.LogManager
import org.apache.log4j.Logger
import org.eclipse.core.internal.resources.ResourceException
import org.eclipse.core.resources.IFile
import org.eclipse.core.resources.IProject
import org.eclipse.core.resources.IResource
import org.eclipse.core.resources.IResourceProxy
import org.eclipse.core.resources.IResourceProxyVisitor
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.Assert
import org.eclipse.core.runtime.CoreException
import org.eclipse.core.runtime.IProgressMonitor
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.core.runtime.Path
import org.eclipse.core.runtime.Status
import org.eclipse.core.runtime.jobs.Job
import org.eclipse.emf.common.command.Command
import org.eclipse.emf.common.util.BasicEMap
import org.eclipse.emf.common.util.EMap
import org.eclipse.emf.common.util.SegmentSequence
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.common.util.WrappedException
import org.eclipse.emf.ecore.EAttribute
import org.eclipse.emf.ecore.EDataType
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EStructuralFeature
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.resource.Resource.IOWrappedException
import org.eclipse.emf.ecore.resource.ResourceSet
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.emf.ecore.xmi.XMLResource
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl
import org.eclipse.emf.edit.command.SetCommand
import org.eclipse.emf.edit.domain.EditingDomain
import org.eclipse.emf.edit.domain.IEditingDomainProvider
import org.eclipse.jface.dialogs.MessageDialog
import org.eclipse.swt.widgets.Display
import org.eclipse.ui.IEditorPart
import org.eclipse.ui.IEditorReference
import org.eclipse.ui.PlatformUI
import org.eclipse.ui.part.FileEditorInput
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.ui.IWorkbenchPage
import org.eclipse.ui.PartInitException
import org.eclipse.ui.IWorkbench

class ComponentAPI {

	static Logger logger = LogManager.getLogger(Utils);

	static ComponentAPI eInstance = init();

	static def ComponentAPI getInstance() {
		return eInstance;
	}

	static def ComponentAPI init() {
		new ComponentAPI
	}

	/**
	 * **************************************************************************************************************
	 * Cache of enabled configurators and exported features
	 */
	static EMap<String, Boolean> enabledConfigurators = new BasicEMap<String, Boolean>();

	def static adaptEnabledConfigurator(String uri, boolean b) {
		enabledConfigurators.put(uri, b)
	}

	def static boolean isEnabled(EObject obj, String prefix) {
		if (obj.eResource === null) {
			return false
		}
		val app = ComponentAPI.getApplication(obj.eResource)
		if (app === null || app.eResource() === null) {
			return false
		}
		val appFile = Utils.getFileFromResource(app.eResource().getURI());
		val project = appFile.getProject();
		val Boolean enabled = enabledConfigurators.get(project.name + "/" + prefix);
		if (enabled !== null) {
			return enabled.booleanValue
		}
		false
	}

	def static boolean isEnabled(EObject obj) {
		if(isBeingValidated) return false;

		if(isCommandBeingEvaluated) return false;

		if (obj.eResource === null) {
			return false
		}
		val uri = EcoreUtil.getURI(obj).toString
		val pos = uri.indexOf('#')
		val Boolean enabled = enabledConfigurators.get(uri.substring(0, pos));
		if (enabled !== null) {
			return enabled.booleanValue
		}
		false
	}

	def static boolean isEnabled(String uri) {
		val pos = uri.indexOf('#')
		var uriRoot = uri
		if (pos > 0) {
			uriRoot = uri.substring(0, pos)
		}
		val Boolean enabled = enabledConfigurators.get(uriRoot);
		if (enabled !== null) {
			return enabled.booleanValue
		}
		false
	}

	/**
	 * Various caches used to handle applications configurators' states
	 */
	Map<String, EditingDomain> editingDomains = newHashMap

	static def void addEditingDomain(String nsPrefix, EditingDomain domain) {

		if (nsPrefix === null || nsPrefix.isEmpty) {
			logger.error('''cannot add editing domain for: '«nsPrefix»' ''')
		}
		eInstance.editingDomains.put(nsPrefix, domain)
	}

	static def void removeEditingDomain(String nsPrefix) {
		if (nsPrefix === null || nsPrefix.isEmpty) {
			logger.error('''cannot remove editing domain for: '«nsPrefix»' ''')
		}
		eInstance.editingDomains.put(nsPrefix, null)
	}

	static def EditingDomain getEditingDomain(String nsPrefix) {
		if (nsPrefix === null || nsPrefix.isEmpty) {
			logger.error('''cannot remove editing domain for: '«nsPrefix»' ''')
		}
		eInstance.editingDomains.get(nsPrefix)
	}

	/**/
	/* ************************************************************************************************************** */
	/**
	 * **************************************************************************************************************
	 * Cache of applications 
	 */
	static EMap<IProject, Application> applications = new BasicEMap<IProject, Application>();

	/**
	 * Get the appllication related to a resource being edited
	 * Returns the Application (create if it does not exist yet)
	 */
	static def Application getApplication(Resource r) {
		logger.setLevel(Utils.debugOption ? Level.DEBUG : Level.INFO);
		if (r === null) {
			return null
		}
		val ifile = Utils.getFileFromResource(r.URI)
		val project = ifile.project

		// First get it from the cache
		var application = applications.get(project)
		if (application === null) {
			// If not found try to load it from the 'application.appcfg' file in the current project
			application = getApplicationFromFile(project)
			if (application === null) {
				// If still not found create it
				application = createApplication(r)
			}
			applications.put(project, application);
			refreshApplication(application)
		}
		return application
	}

	/**
	 * Load the application from its file inside the project
	 * NB: there is only one file named application./appcfg inside a project 
	 */
	static def private Application getApplicationFromFile(IProject project) {

		val applications = Utils.findFilesWithExtension(project, "appcfg")
		if (applications.size == 0) {
			return null
		}
		if (applications.size > 1) {
			logger.error("Must have only one appcfg file inside a project")
		}
		val appliFile = applications.get(0)

		val path = appliFile.fullPath
		val loadUri = URI.createPlatformResourceURI(path.toString, true)
		val a = getEObject(loadUri.appendFragment("application"))
		if (Application.isInstance(a)) {
			val app = Application.cast(a)
			return app
		}
		return null
	}

	/**
	 * Cache of configurators associated to an application (identified by its URI)
	 * Sub cache contains the configurators, identified by their uri
	 */
	static EMap<String, EMap<String, Component>> applicationConfigurators = new BasicEMap<String, EMap<String, Component>>();

	/**
	 * Create an application file inside a project 
	 */
	static def createApplication(Resource r) {
		// Create a resource set
		val resourceSet = new ResourceSetImpl();

		// Create a resource for this file.
		val ifile = Utils.getFileFromResource(r.URI)
		val project = ifile.project
		val filePath = "application.appcfg";
		val appFile = project.getFile(new Path(filePath));
		val fileURI = URI.createPlatformResourceURI(appFile.getFullPath().toString(), true);
		val resource = resourceSet.createResource(fileURI);

		// Create the application and populate it
		val app = ComponentPackage.eINSTANCE.getComponentFactory().createApplication
		app.name = "application"

		val fileName = r.URI.lastSegment
		val suffix = fileName.substring(fileName.indexOf('.') + 1)

		// Create one ComponentReference object for all found configurator inside the project
		for (URI c : getConfigurators(r, appFile, suffix)) {
			val ComponentReference ref = ComponentPackage.eINSTANCE.getComponentFactory().createComponentReference();
			ref.setUri(c.toFileString());
			val name = "/" + project.name + "/" + c
			val objectUri = URI.createPlatformResourceURI(name, true)
			val compObj = getEObject(objectUri.appendFragment(""))
			val component = Component.cast(compObj)
			addConfiguratorToApplication(resource.toString, component)

			ref.enabled = true
			app.getComponents().add(ref);
			adaptEnabledConfigurator(objectUri.toString, true)
		}
		if (app !== null) {
			resource.getContents().add(app);
		}

		// Save the contents of the resource to the file system.
		val options = new HashMap<Object, Object>();
		options.put(XMLResource.OPTION_ENCODING, "UTF-8");
		try {
			resource.save(options);
		} catch (IOWrappedException e) {
			e.printStackTrace
		}

		// Add the project's application into the cache
		applications.put(project, app);
		return app
	}

	/**
	 * Refresh the content of the application (adapt enabled and disabled configurators)
	 */
	static def refreshApplication(Application application) {

		synchronized (eInstance) {
			val appFile = Utils.getFileFromResource(application.eResource().getURI())
			val project = appFile.getProject()

			val appUri = application.eResource().getURI().path()
			var componentsUri = applicationConfigurators.get(appUri)
			if (componentsUri === null) {
				componentsUri = new BasicEMap<String, Component>();
				applicationConfigurators.put(appUri, componentsUri)
			}

			// reset all application's enabled components
			for (en : Collections.unmodifiableList(enabledConfigurators.keySet.toList)) {
				val name = "/" + project.name + "/"
				val projectUri = URI.createPlatformResourceURI(name, true)
				if (en.startsWith(projectUri.toString)) {
					enabledConfigurators.removeKey(en)
					componentsUri.removeKey(en)
				}
			}

			for (ComponentReference ref : application.components) {
				val name = "/" + project.name + "/" + ref.uri
				val objectUri = URI.createPlatformResourceURI(name, true)
				setEnabledConfigurator(application, objectUri.toString, ref.enabled)
			}
		}
	}

	/**
	 * Refresh the application in the project identified by a contained resource (for example
	 * a Can configurator's resource)
	 */
	static def refreshApplication(Resource r) {
		if (r === null) {
			return null
		}
		val ifile = Utils.getFileFromResource(r.URI)
		val project = ifile.project

		// Clears the cache
		applicationConfigurators.clear
		refreshApplication(project)
	}

	/**
	 *  Refresh the application in the project
	 */
	static def refreshApplication(IProject project) {
		refreshApplication(project, true)
	}

	/**
	 * Refresh the application associated to a project, and create it if needed
	 */
	static def void refreshApplication(IProject project, boolean createIfNeeded) {
		val applicationList = Utils.findFilesWithExtension(project, "appcfg")
		var Application application = null

		// Must have exactly one application file per project
		if (applicationList.size == 0) {
			if (createIfNeeded) {
				val resourceSet = new ResourceSetImpl()
				resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put(
					Resource.Factory.Registry.DEFAULT_EXTENSION, new XMIResourceFactoryImpl());

				// Register the package to ensure it is available during loading.
				//
				resourceSet.getPackageRegistry().put(ComponentPackage.eNS_URI, ComponentPackage.eINSTANCE);

				val r = resourceSet.createResource(
					URI.createPlatformResourceURI("/" + project.name + "/application.appcfg", true))
				application = createApplication(r)
				if (application === null) {
					return
				}
			} else {
				return
			}
		} else if (applicationList.size > 1) {
			logger.error("Must have only one appcfg file inside a project")
			return
		} else {
			// Load application file from its resource
			val appliFile = applicationList.get(0)
			val path = appliFile.fullPath
			val loadUri = URI.createPlatformResourceURI(path.toString, true)
			val a = getEObject(loadUri.appendFragment("application"))
			if (Application.isInstance(a)) {
				application = Application.cast(a)
			}
		}

		if (application === null) {
			return
		}
		val r = application.eResource

		val filePath = "application.appcfg";
		val appFile = project.getFile(new Path(filePath));

		var toBeSaved = false
		for (URI c : getConfigurators(r, appFile)) {
			val old = application.components.findFirst[it.uri.equals(c.toFileString())]
			if (old === null) {
				// Add uri only if it does not exist yet
				val ComponentReference ref = ComponentPackage.eINSTANCE.getComponentFactory().
					createComponentReference();
				ref.setUri(c.toFileString());
				ref.enabled = true
				application.getComponents().add(ref);

				val name = "/" + project.name + "/" + c
				val objectUri = URI.createPlatformResourceURI(name, true)
				setEnabledConfigurator(application, objectUri.toString, true)
				toBeSaved = true
			}
		}
		val toBeRemoved = newArrayList
		for (comp : application.components) {
			if (!checkUriExists(project, comp.uri)) {
				toBeRemoved.add(comp)
				toBeSaved = true
			}
		}
		val count = toBeRemoved.size
		val originalSize = application.components.size
		for (comp : toBeRemoved) {
			if (!application.components.remove(comp)) {
				println('''Cannot remove «comp.uri» from project «project.name»''')
			}
		}
		Assert.isTrue(application.components.size + count ==
			originalSize, '''Expecting «originalSize» == «application.components.size» + «count»''')

		if (toBeSaved) {
			applications.remove(project);
			Display.getDefault().asyncExec(new Runnable() {
				override void run() {
					// Save the contents of the resource to the file system.
					val options = new HashMap<Object, Object>();
					options.put(XMLResource.OPTION_ENCODING, "UTF-8");
					r.save(options);
				}
			});

			// refresh project content in the workspace
			project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
		}
		applications.put(project, application)
		refreshEnabledConfigurators(application);
	}

	/**
	 *  Add resources: need to refresh the caches
	 */
	static def removeResources(Set<IResource> toBeRemoved) {
		try {
			val applications = new HashMap<IProject, Integer>();
			val resources = new HashMap<IProject, Set<IResource>>();
			logger.debug("Remove: " + toBeRemoved.map[name].join(","))
			for (r : toBeRemoved) {
				val project = r.project
				var size = 0
				if (project !== null && project.isAccessible) {
					for (a : applicationConfigurators.keySet) {
						if (a.contains(project.name)) {
							size = applicationConfigurators.get(a).keySet.size
							applications.put(project, size)
						}
					}
					var projectResources = resources.get(project)
					if (projectResources === null) {
						projectResources = new HashSet<IResource>();
						resources.put(project, projectResources)
					}
					projectResources.add(r)
				}
			}

			DerivedAttributesNotifier.clearExternalAttributes
			applicationConfigurators.clear
			applicationEObjects.clear
			for (p : resources.keySet) {

				val size = applications.get(p)?:0
				var nbRemovedFiles = 0

				var appliRemoved = false
				for (r : resources.get(p)) {
					// Get the workspace-relative path as a string
					val workspaceRelativePath = r.fullPath.toString

					// Create the EMF URI
					val emfUri = URI.createPlatformResourceURI(workspaceRelativePath, true);
					logger.debug('''Remove resource «emfUri»''')
					if (workspaceRelativePath.contains("application.appcfg")) {
						appliRemoved = true;
					}
					nbRemovedFiles++
				}
				if (size - nbRemovedFiles > 0) {
					refreshApplication(p, appliRemoved)
				}
			}

		} catch (ResourceException re) {
			logger.error(re.message)
		}
	}

	static class ProjectListVisitor implements IResourceProxyVisitor {

		List<IResource> _list;

		new(List<IResource> list) {
			_list = list;
		}

		override visit(IResourceProxy proxy) throws CoreException {
			if (proxy.getType() == IResource.FILE) {
				_list.add(proxy.requestResource)
			}
			return true;
		}
	}

	def static int projectNbFiles(IProject project) {
		val files = new ArrayList<IResource>();
		val visitor = new ProjectListVisitor(files);
		try {
			project.accept(visitor, IResource.NONE);
		} catch (CoreException e) {
			logger.error(e.getMessage());
		}
		files.size
	}

	static private def displayCaches() {
		DerivedAttributesNotifier.displayCaches();

		var indent = 1;
		System.out.println("\t".repeat(indent) + "Configurators");
		indent++
		for (cfg : enabledConfigurators.keySet()) {
			val externalAttrs = enabledConfigurators.get(cfg);
			System.out.println("\t".repeat(indent) + cfg + " -> " + externalAttrs.toString);
		}
		indent--

//		System.out.println("\t".repeat(indent) + "Application configurators");
//		indent++
//		for (cfg : applicationConfigurators.keySet()) {
//			val externalAttrs = applicationConfigurators.get(cfg);
//			for (k : externalAttrs.keySet()) {
//				val obj = externalAttrs.get(k)
//				if( obj !== null) {
//					System.out.println("\t".repeat(indent) + cfg + " -> " + obj.toString);
//				}
//			}
//		}
//		indent--
//
//		System.out.println("\t".repeat(indent) + "Application objects");
//		indent++
//		for (cfg : applicationEObjects.keySet()) {
//			val externalAttrs = applicationEObjects.get(cfg);
//			for (k : externalAttrs.keySet()) {
//				val obj = externalAttrs.get(k)
//				if( obj !== null) {
//					val str = obj.getClass().getName() + "@" + Integer.toHexString(obj.hashCode())
//					System.out.println("\t".repeat(indent) + cfg + " -> " + str);
//				}
//			}
//		}
//		indent--
	}

	/**
	 *  Add resources: need to refresh the caches
	 */
	static def addResources(Set<IResource> toBeAdded) {
		val resources = new HashMap<IProject, Set<IResource>>();
		logger.debug("Add: " + toBeAdded.map(it|it.name).join(","))
		for (r : toBeAdded) {
			val project = r.project
			if (project !== null && project.isAccessible) {
				var projectResources = resources.get(project)
				if (projectResources === null) {
					projectResources = new HashSet<IResource>();
					resources.put(project, projectResources)
				}
				projectResources.add(r)
			}
		}

		for (p : resources.keySet) {

			for (r : resources.get(p)) {
				// Get the workspace-relative path as a string
				val workspaceRelativePath = r.fullPath.toString

				// Create the EMF URI
				val emfUri = URI.createPlatformResourceURI(workspaceRelativePath, true);
				logger.debug('''Add resource «emfUri»''')
			}
			refreshApplication(p, true)
		}
	}

	/**
	 *  Add resources: need to refresh the caches
	 */
	static def changeResources(Set<IResource> toBeChanged) {
		val resources = new HashMap<IProject, Set<IResource>>();
		logger.debug("Change: " + toBeChanged.map(it|it.name).join(","))
		for (r : toBeChanged) {
			val project = r.project
			if (project !== null && project.isAccessible) {
				var projectResources = resources.get(project)
				if (projectResources === null) {
					projectResources = new HashSet<IResource>();
					resources.put(project, projectResources)
				}
				projectResources.add(r)
			}
		}

		val resourceSet = new ResourceSetImpl();
		for (p : resources.keySet) {

			val app = getApplicationFromFile(p)
			for (r : resources.get(p)) {
				// Get the workspace-relative path as a string
				val workspaceRelativePath = r.fullPath.toString

				// Create the EMF URI
				val emfUri = URI.createPlatformResourceURI(workspaceRelativePath, true);

				try {
					val resource = resourceSet.getResource(emfUri, false);
					if (resource !== null) {
						logger.debug('''Change resource «emfUri»''')
					}
				} catch (ResourceException e) {
					// ignore error
				}
			}
			ComponentAPI.refreshExternalDependencies(app);
			refreshApplication(p, false)
		}
	}

	/* ************************************************************************************************************** */
	/**
	 * Check whether an URI exists inside a project (i.e. points to a valid file) 
	 */
	def static boolean checkUriExists(
		IProject project,
		String uriStr
	) {
		val workspaceRoot = ResourcesPlugin.getWorkspace().getRoot()
		val path = new Path("/" + project.getName() + "/" + uriStr.toString());
		val f = workspaceRoot.getFile(path);
		// println('''file «f» «f.exists?"exists":"does not exist"»''')
		return f.exists
	}

	/**
	 * Get a configurator from its configurator Id (i.e. the plugin id providing its model)
	 */
	static def Component getConfigurator(EObject object, String confId) {
		// Get the application in which the configurator must be found
		val application = getApplication(object.eResource)
		if (application !== null) {
			val comp = getConfigurator(application, confId)
			return comp
		} else {
			return null
		}
	}

	/**
	 * Get a configurator from its configurator Id (i.e. the plugin id providing its model)
	 */
	static def Component getConfigurator(Application app, String compId) {
		if (app === null) {
			return null
		}
		if (app.eResource === null) {
			return null
		}
		val appUri = app.eResource().getURI().path()
		var configuratorsUri = applicationConfigurators.get(appUri)
		if (configuratorsUri === null) {
			configuratorsUri = new BasicEMap<String, Component>();
			applicationConfigurators.put(appUri, configuratorsUri)
		}

		val appFile = Utils.getFileFromResource(app.eResource.URI)
		val project = appFile.project
		val application = app
		val configurators = application.getComponents();

		// Traverse all configurators until we found an enabled matching configurator
		var Component found = null;
		for (c : configurators) {
			if (found === null) {
				val name = "/" + project.name + "/" + c.uri
				val objectUri = URI.createPlatformResourceURI(name, true)
				if (isEnabled(objectUri.toString)) {
					var EObject comp = configuratorsUri.get(objectUri.toString)
					if (comp === null) {
						comp = getEObject(application, objectUri.appendFragment(""))
						configuratorsUri.put(objectUri.toString, comp as Component)
					}
					if (Component.isInstance(comp)) {
						val ccomp = Component.cast(comp)
						addConfiguratorToApplication(appUri, ccomp)
						if (ccomp.exportedFeatures.contains(compId)) {
							found = ccomp
						}
					}
				}
			}
		}
		if (found !== null) {
			return found
		} else {
			return configuratorsUri.get(compId)
		}
	}

	/**
	 * Add a configurator to the configurators managed by an application
	 */
	def static addConfiguratorToApplication(String appUri, Component app) {

		val appFile = Utils.getFileFromResource(app.eResource.URI);
		val project = appFile.project
		val projectPath = "/" + project.name + "/"

		var appConfigurators = applicationConfigurators.get(appUri)
		if (appConfigurators === null) {
			appConfigurators = new BasicEMap<String, Component>()
			applicationConfigurators.put(appUri, appConfigurators)
		}
		for (f : Utils.removeDuplicates(app.exportedFeatures)) {
			appConfigurators.put(projectPath + f, app)
		}
		appConfigurators.put(app.eResource.URI.toString, app)
	}

// Load an object from its URI
	static def EObject getEObject(URI eObjectURI) {
		return eInstance.loadEObject(eObjectURI);
	}

	/**
	 * **********************************************************************************************************
	 * Cache of application objects
	 */
	static EMap<String, EMap<String, EObject>> applicationEObjects = new BasicEMap<String, EMap<String, EObject>>();

	/**
	 * Get an object from its URI inside the desired application
	 */
	static def EObject getEObject(Application app, URI eObjectURI) {
		val appUri = app.eResource().getURI().path()

		var eobjectsUri = applicationEObjects.get(appUri)
		if (eobjectsUri === null) {
			eobjectsUri = new BasicEMap<String, EObject>();
			applicationEObjects.put(appUri, eobjectsUri)
		}
		var o = eobjectsUri.get(eObjectURI.toString)
		if (o === null) {
			o = eInstance.loadEObject(eObjectURI);
			eobjectsUri.put(eObjectURI.toString, o)
		}
		return o
	}

	/* ********************************************************************************************************** */
	private def EObject loadEObject(URI eObjectURI) {
		var ResourceSet resourceSet = null
		if (resourceSet === null) {
			resourceSet = new ResourceSetImpl();
			resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put(
				Resource.Factory.Registry.DEFAULT_EXTENSION, new XMIResourceFactoryImpl());

			// Register the package to ensure it is available during loading.
			//
			resourceSet.getPackageRegistry().put(ComponentPackage.eNS_URI, ComponentPackage.eINSTANCE);
		}

		// Load the resource if necessary
		try {

			try {
				val resource = resourceSet.getResource(eObjectURI.trimFragment(), true);
				// Retrieve the EObject from the resource
				val eObject = resourceSet.getEObject(eObjectURI, true);
				if (eObject === null) {
					if (resource !== null) {
						println("load: " + eObjectURI)
						resource.load(resourceSet.getLoadOptions());

						val contents = resource.getContents();
						if (!contents.isEmpty) {

							val comp = contents.get(0);
							if (Component.isInstance(comp)) {
								return comp
							}
						}
					}
				} else {
					return eObject;
				}
			} catch (ResourceException re) {
				return null;
			}
		} catch (WrappedException ioe) {
			return null;
		}

	}

	/**
	 * Return the component edited by an active editor, using a component of the same class
	 * 
	 * @param root The root element from which to start the search
	 * 
	 * @return The value of the property or null if not edited yet
	 */
	static def Component getConfiguratorProperty(Component root) {
		var newRoot = root
		if(root === null) return null;

		// Get editing domain from the cache
		val editingDomain = eInstance.editingDomains.get(getPrefix(root));
		if (editingDomain !== null) {
			val rootUri = EcoreUtil.getURI(newRoot)
			newRoot = getEObjectFromEditingDomain(editingDomain, rootUri) as Component
			return newRoot
		}
		return null
	}

	/**
	 * Return the component edited by an active editor, using a component of the same class
	 * 
	 * @param root The root element from which to start the search
	 * 
	 * @return The value of the property
	 */
	static def Application getEditedApplication(Application root) {
		var newRoot = root
		if (root === null) {
			return null;
		}
		val editingDomain = eInstance.editingDomains.get(getPrefix(root));
		if (editingDomain !== null) {
			val rootUri = EcoreUtil.getURI(newRoot)
			newRoot = getEObjectFromEditingDomain(editingDomain, rootUri) as Application
			return newRoot
		}
	}

	static final String PATH_SEPARATOR = "/"; // $NON-NLS-1$
	static final String SEGMENT_SEPARATOR = "@"; // $NON-NLS-1$

	/**
	 * Get a property's value from its path starting from root component object
	 * 
	 * @param root The root element from which to start the search
	 * @param path  The path to the property to be found (this is a slash ('/')
	 *              separated list of attributes e.g.:
	 *              drive_management/drive_settings/torque_and_flux_regulators/manual_editing_enabled
	 * 
	 * @return The value of the property
	 */
	static def Pair<Component, Object> getPropertyValueFromPath(
		Component root,
		String path
	) {
		var newRoot = root
		if (path.contains(PATH_SEPARATOR)) {
			var Object result = null

			// Format the sequence of properties to be given to the EcoreUtil.getEObject()
			// method which expects paths starting with '@'
			val pathSegments = SegmentSequence.create(PATH_SEPARATOR, path).segments()
			var segments = new ArrayList<String>()
			var groupSegment = "";
			for (var i = 0; i < pathSegments.length - 1; i++) {
				val seg = SEGMENT_SEPARATOR + pathSegments.get(i)
				segments.add(seg)
				if (groupSegment.length() > 0) {
					groupSegment += PATH_SEPARATOR + seg
				} else {
					groupSegment += seg
				}
			}
			val propName = pathSegments.get(pathSegments.length - 1)

			// Get the group containing the property, get its feature and return its value
			try {
				var group = EcoreUtil.getEObject(newRoot, groupSegment)
				val property = group.eClass().getEStructuralFeature(propName)
				if (property !== null) {
					result = group.eGet(property)
					val editingDomain = eInstance.editingDomains.get(getPrefix(root));
					if (editingDomain !== null) {
						val groupUri = EcoreUtil.getURI(group)
						group = getEObjectFromEditingDomain(editingDomain, groupUri) as EObject
						result = group.eGet(property)
					}
				} else {
					val editingDomain = eInstance.editingDomains.get(getPrefix(root));
					if (editingDomain !== null) {
						val groupUri = EcoreUtil.getURI(group)
						group = getEObjectFromEditingDomain(editingDomain, groupUri) as EObject
						result = EcoreUtil.getEObject(newRoot, propName);
					// println(result)
					}
				}
			} catch (IllegalArgumentException iae) {
				logger.error("Cannot find property named '" + path + "'")
			}
			return new Pair(newRoot, result)
		} else {
			val property = newRoot.eClass().getEStructuralFeature(path)
			if (property !== null) {
				var propertyValue = newRoot.eGet(property)

				val editingDomain = eInstance.editingDomains.get(getPrefix(root));
				if (editingDomain !== null) {
					val rootUri = EcoreUtil.getURI(newRoot)
					newRoot = getEObjectFromEditingDomain(editingDomain, rootUri) as Component
					if (newRoot !== null) {
						propertyValue = newRoot.eGet(property)
					} else {
						propertyValue = null
					}
				}
				return new Pair(newRoot, propertyValue)
			} else {
				logger.error("Cannot find property named '" + path + "'")
				return null
			}
		}
	}

	/**
	 * Get a property's value from its path starting from root component object
	 * 
	 * @param root The root element from which to start the search
	 * @param path  The path to the property to be found (this is a slash ('/')
	 *              separated list of attributes e.g.:
	 *              drive_management/drive_settings/torque_and_flux_regulators/manual_editing_enabled
	 * 
	 * @return The value of the property
	 */
	static def EStructuralFeature getFeatureFromPath(
		Component root,
		String path
	) {
		var newRoot = root
		if (path.contains(PATH_SEPARATOR)) {
			// Format the sequence of properties to be given to the EcoreUtil.getEObject()
			// method which expects paths starting with '@'
			val pathSegments = SegmentSequence.create(PATH_SEPARATOR, path).segments()
			var segments = new ArrayList<String>()
			var groupSegment = "";
			for (var i = 0; i < pathSegments.length - 1; i++) {
				val seg = SEGMENT_SEPARATOR + pathSegments.get(i)
				segments.add(seg)
				if (groupSegment.length() > 0) {
					groupSegment += PATH_SEPARATOR + seg
				} else {
					groupSegment += seg
				}
			}
			val propName = pathSegments.get(pathSegments.length - 1)

			// Get the group containing the property, get its feature and return its value
			try {
				var group = EcoreUtil.getEObject(newRoot, groupSegment)
				val property = group.eClass().getEStructuralFeature(propName)
				return property
			} catch (IllegalArgumentException iae) {
				logger.error("Cannot find property named '" + path + "'")
				return null
			}
		} else {
			val property = newRoot.eClass().getEStructuralFeature(path)
			if (property !== null) {
				return property
			} else {
				logger.error("Cannot find property named '" + path + "'")
				return null
			}
		}
	}

	static def EAttribute getEAttributeFromURI(String uriString) {
		// Create a ResourceSet to manage resources
		val resourceSet = new ResourceSetImpl();

		// Create a URI from the string
		val uri = URI.createURI(uriString);

		// Load the resource
		val resource = resourceSet.getResource(uri.trimFragment(), true);

		// Get the EObject from the fragment
		val eObject = resource.getEObject(uri.fragment());

		if (eObject instanceof EAttribute) {
			return eObject as EAttribute
		}

		return null; // Not an EAttribute or not found
	}

	def static getPrefix(EObject root) {
		if (root !== null && root.eResource !== null) {
			val fileUri = root.eResource.URI
			return fileUri.toString()
		}
		return null
	}

	def static String getRelativePath(EObject eObject) {
		if (eObject === null) {
			return null;
		}

		val uri = EcoreUtil.getURI(eObject);
		return uri.fragment();
	}

	def static EObject getEObjectFromEditingDomain(EditingDomain editingDomain, URI eObjectURI) {
		var rs = editingDomain.getResourceSet()
		val resource = rs.getResource(eObjectURI.trimFragment(), true);
		if (resource !== null) {
			return resource.getEObject(eObjectURI.fragment());
		}
		return null;
	}

	static class ProjectVisitor implements IResourceProxyVisitor {

		List<IResource> _list;
		String _extensions;

		new(List<IResource> list, String ext) {
			_list = list;
			_extensions = (ext === null ? "" : ext)
		}

		override visit(IResourceProxy proxy) throws CoreException {
			if (proxy.getType() == IResource.FILE) {
				val path = new Path(proxy.getName())
				if (checkComponent(path)) {
					_list.add(proxy.requestResource)
				}
			}
			return true;
		}

		def boolean checkComponent(Path path) {
			for (e : _extensions.split(",")) {
				if (path.toString.endsWith(e)) {
					return true;
				}
			}
		}
	}

	static private def Resource getResource(Resource context, String uriStr) {
		var Resource res = null;
		if (uriStr !== null && context !== null) {
			try {
				val r = EcoreUtil2.getResource(context, uriStr);
				if (r !== null) {
					r.load(null);
				}
				if (r === null) {
					return null;
				}
				if (r.getContents().size() > 0) {
					res = r;
				}

			} catch (Exception e) {
				logger.error(e.getMessage());
			}
		}
		return res;
	}

	static def Collection<URI> getConfigurators(Resource resource, IFile modelFile) {
		val list = new ArrayList<URI>();
		val localModels = getLocalModels(resource, modelFile);
		for (URI u : localModels) {
			list.add(u);
		}
		return list;
	}

	static def Collection<URI> getConfigurators(Resource resource, IFile modelFile, String suffix) {
		val list = new ArrayList<URI>();
		val localModels = getLocalModels(resource, modelFile, suffix);
		for (URI u : localModels) {
			list.add(u);
		}
		return list;
	}

	static private def Collection<URI> getLocalModels(Resource context, IFile resource) {
		return getLocalModels(context, resource, null)
	}

	static private def Collection<URI> getLocalModels(Resource context, IFile resource, String suffix) {
		val files = new ArrayList<IResource>();

		val resourceSet = new ResourceSetImpl();
		val file = resource;
		if (file === null) {
			return new ArrayList<URI>();
		}
		val project = file.getProject();
		val visitor = new ProjectVisitor(files, suffix);
		val handles = new ArrayList<URI>();
		try {
			project.accept(visitor, IResource.NONE);
		} catch (ResourceException e) {
			logger.error(e.getMessage());
			return handles;
		} catch (CoreException e) {
			logger.error(e.getMessage());
			return handles;
		}
		val iter = files.iterator();
		while (iter.hasNext()) {
			val f = iter.next() as IFile;
			var String decodedUri = null
			try {
				// println(f.getProjectRelativePath().toOSString());
				var match = true
				if (suffix !== null) {
					match = f.getProjectRelativePath().toOSString().endsWith(suffix)
				}
				if (match) {
					decodedUri = URLDecoder.decode(f.getProjectRelativePath().toOSString(), "UTF-8");
					// println(decodedUri)
					var loadResource = getResource(context, decodedUri);
					val resource2 = resourceSet.createResource(URI.createFileURI(decodedUri));
					resourceSet.resources.add(resource2)
//					var loadResource = getResource(resource2, decodedUri);
//					var loadResource = EcoreUtil2.getResource(resource2, decodedUri);
					if (loadResource !== null) {
						loadResource.load(resourceSet.getLoadOptions());
						val contents = loadResource.getContents();
						if (contents.size() > 0) {
							val comp = contents.get(0);
							// println(comp)
							if (Component.isInstance(comp)) {
								val applicationComp = Component.cast(comp)
								// println(applicationComp.name)
								if (applicationComp.isInApplicationModel) {
									val uri = URI.createFileURI(decodedUri);
									handles.add(uri);
								}
							}
						}
//					} else {
//						val fileURI2 = URI.createFileURI(decodedUri)
//						val resource2 = resourceSet.createResource(fileURI2);
//						loadResource = EcoreUtil2.getResource(resource2, fileURI2.toFileString);
//						
//						try {
//							resource2.load(resourceSet.getLoadOptions());
//							if (!resource2.getContents().isEmpty()) {
//								val root2 = resource2.getContents().get(0)
//								System.out.println("Root object: " + root2)
//							}
//							if (loadResource !== null) {
//								loadResource.load(resourceSet.getLoadOptions());
//								val contents = loadResource.getContents();
//								if (contents.size() > 0) {
//									val comp = contents.get(0);
//									// println(comp)
//									if (Component.isInstance(comp)) {
//										val applicationComp = Component.cast(comp)
//										println(applicationComp.name)
//										if (applicationComp.isInApplicationModel) {
//											val uri = URI.createFileURI(decodedUri);
//											handles.add(uri);
//										}
//									}
//								}
//							}
//						} catch (Exception e) {
//						}
//					}
//				}
					}
				}
			} catch (Exception e) {
				println(e.message)
			}
		}
		return handles;
	}

	static def EditingDomain getEditingDomainFromEditor(IEditorPart editorPart) {
		// Check if the editor part implements IEditingDomainProvider
		if (editorPart instanceof IEditingDomainProvider) {
			return (IEditingDomainProvider.cast(editorPart)).getEditingDomain();
		}

		// Check if the editor part has an adapter that provides the EditingDomain
		val editingDomain = editorPart.getAdapter(EditingDomain) as EditingDomain;
		if (editingDomain !== null) {
			return editingDomain;
		}

		// If no EditingDomain is found, return null
		return null;
	}

	static def EditingDomain getEditingDomain(EObject obj) {
		if (obj === null) {
			return null
		}

		val isWorkbenchRunning = PlatformUI.isWorkbenchRunning();
		if (!isWorkbenchRunning) {
			return null
		}
		val workbench = PlatformUI.getWorkbench()
		val workbenchWindow = workbench.getActiveWorkbenchWindow()
		if (workbenchWindow === null) {
			return null
		}
		val page = workbenchWindow.getActivePage()

		if (page === null) {
			return null
		}

		val IEditorReference[] editorReferences = page.getEditorReferences()

		for (IEditorReference editorReference : editorReferences) {
			try {
				val editorPart = editorReference.getEditor(true)
				if (editorPart !== null && editorPart.getEditorInput() instanceof FileEditorInput) {
					val fileEditorInput = FileEditorInput.cast(editorPart.getEditorInput())
					val resource = obj.eResource();
					val fileUri = fileEditorInput.URI
					if (resource !== null) {
						val fileUri2 = URI.createFileURI(resource.getURI().toPlatformString(true))
						if (fileUri.toString.contains(fileUri2.toString)) {
							val editingDomain = getEditingDomainFromEditor(editorPart)
							if (editingDomain !== null) {
								return editingDomain
							}
						}
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return null
	}

	static def boolean isBeingValidated() {
		for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
			if (element.getClassName().startsWith("org.eclipse.emf.ecore.util.EObjectValidator")) {
				if (element.methodName.equals("validate")) {
					return true;
				}
			}
		}
		return false;
	}

	static def boolean isCommandBeingEvaluated() {
		for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
			if (element.getClassName().startsWith("org.eclipse.emf.common.command.AbstractCommand")) {
				if (element.methodName.equals("canExecute")) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * @param object the caller object
	 * @param attr the attribute to be set with the result value
	 * @param configuratorId the plugin id to the external configurator containing the expected attribute
	 * @param path the path to the attribute inside the external configurator
	 * @param oldValue the previous value (the one originally stored inside the component)
	 * @param previousComponent the previous component
	 * 
	 * @returns an attribute value object with the value and the Component from which this value comes
	 */
	def static ExternalAttributeValueResult getExternalAttributeValue(EObject object, EAttribute attr,
		String configuratorId, String path, Object oldValue, Component previousComponent) {
		if (isBeingValidated) {
			return new ExternalAttributeValueResult(oldValue, previousComponent);
		}

		// Let C1 be the component that uses an attribute from another component C2
		// if external value is edited in C2 editor {
		// - need to compare old value stored inside the C1 file and the value inside C2 editor
		// - if different must display a confirmation dialog
		// - each time the value is changed inside C2 editor, the change must be reflected in C1
		// } else {
		// - need to compare old value stored inside the C1 file and the value inside C2 editor
		// - if different must display a confirmation dialog
		// - 
		// }
		val app = ComponentAPI.getApplication(object.eResource)

		val needPopup = previousComponent === null && PlatformUI.workbenchRunning

		// Get external component root object
		var retComponent = ComponentAPI.getConfigurator(app, configuratorId)
		if (retComponent === null) {
			logger.error('''Cannot obtain value from «configuratorId»/«path»''')
			return new ExternalAttributeValueResult(oldValue, previousComponent);
		}

		val valuePair = ComponentAPI.getPropertyValueFromPath(retComponent, path)
		if (valuePair === null) {
			logger.error('''Cannot obtain value from «configuratorId»/«path»''')
			return new ExternalAttributeValueResult(null, retComponent);
		}
		var Object value = valuePair.value
		retComponent = valuePair.key as Component
		val canObject = object
		if (needPopup && value !== null && value.toString.compareTo(oldValue.toString) > 1) {
			val externalFile = retComponent.eResource.URI.lastSegment
			val currentFile = object.eResource.URI.lastSegment
			// Warns the user that the value has changed from outside
			val oldVal = oldValue
			val newVal = value
			Display.getDefault().asyncExec(new Runnable() {

				override void run() {
					val editingDomain = ComponentAPI.getEditingDomain(canObject)
					if (editingDomain !== null) {
						if (MessageDialog.openConfirm(Display.getDefault().getActiveShell(), "Confirmation", '''
						Attribute '«attr.name»':
						Previous stored value in file '«currentFile»' -> «oldVal» differs from
						current value in file '«externalFile»' -> «newVal»
						
						Do you want to align?''')) {
							eInstance.setAttributeValue(canObject, attr, newVal.toString)
						}
					}
				}
			})
		}

		return new ExternalAttributeValueResult(value, retComponent)
	}

	static class DisplayPopupJob extends Job {

		String currentFile
		Object oldVal
		Object newVal
		EObject object
		EAttribute attr

		new(EObject object, EAttribute attr, Object value, Object oldValue) {
			super("DisplayPopup")
			this.object = object
			this.currentFile = object.eResource.URI.lastSegment
			// Warns the user that the value has changed from outside
			this.oldVal = oldValue
			this.newVal = value
			this.attr = attr
		}

		def setAttributeValue(EObject object, EAttribute attribute, String value) {
			val editingDomain = eInstance.editingDomains.get(getPrefix(EcoreUtil.getRootContainer(object)));
			val eClassifier = attribute.EType
			if (eClassifier instanceof EDataType && editingDomain !== null) {
				val eDataType = eClassifier as EDataType
				val eFactory = eDataType.getEPackage().getEFactoryInstance()
				val newValue = eFactory.createFromString(eDataType, value)
				val Command command = SetCommand.create(editingDomain, object, attribute, newValue);
				logger.debug("Attribute '" + attribute.getName() + "' set to " + newValue.toString());
				editingDomain.getCommandStack().execute(command);
			}
		}

		override protected run(IProgressMonitor monitor) {
			Display.getDefault().syncExec(new Runnable() {

				override void run() {
					if (MessageDialog.openConfirm(Display.getDefault().getActiveShell(), "Confirmation", '''
					Attribute '«attr.name»':
					Previous stored value in file '«currentFile»' -> «oldVal» differs from
					current value -> «newVal»
					
					Do you want to align?''')) {
						setAttributeValue(object, attr, newVal.toString)
					}
				}
			})
			val key = object.eResource.URI.toString + "_" + attr.name
			displayPopupJobs.put(key, null)
			return Status.OK_STATUS;
		}

	}

	static EMap<String, DisplayPopupJob> displayPopupJobs = new BasicEMap<String, DisplayPopupJob>();

	static def void displayPopup(EObject object, EAttribute attr, Object value, Object oldValue) {
		if (value !== null) {
			// display popup only if values are different
			val key = object.eResource.URI.toString + "_" + attr.name
			var job = displayPopupJobs.get(key)
			if (job === null) {
				if (value.toString.compareTo(oldValue.toString) != 0) {
					job = new DisplayPopupJob(object, attr, value, oldValue)
					displayPopupJobs.put(key, job)
					job.schedule(100L)
				}
			} else {
				if (value.toString.compareTo(oldValue.toString) != 0) {
					job.schedule(100L)
				} else {
					job.cancel
				}
			}
		}
	}

	/**
	 * Create and ask for execution of a Set command that modifies an attribute's value
	 * One need to get the editing domain responsible for edition of the object to be modified, in order 
	 * the editor must be modified correctly as for EMF rules
	 */
	def setAttributeValue(EObject object, EAttribute attribute, String value) {
		val editingDomain = eInstance.editingDomains.get(getPrefix(EcoreUtil.getRootContainer(object)));
		val eClassifier = attribute.EType
		if (eClassifier instanceof EDataType) {
			val eDataType = eClassifier as EDataType
			val eFactory = eDataType.getEPackage().getEFactoryInstance()
			val newValue = eFactory.createFromString(eDataType, value)
			val Command command = SetCommand.create(editingDomain, object, attribute, newValue);
			// logger.debug("Attribute '" + attribute.getName() + "' set to " + newValue.toString());
			editingDomain.getCommandStack().execute(command);
		}
	}

	/**
	 * Ask for a dependency analysis for a decicated editor editing domain (this means create
	 * the appropriate rules to be warned whenever a dependant attribute is modified
	 */
	static def EObject analyzeDependencies(EditingDomain domain) {
		logger.info("Analyze dependencies for: " + domain)
		val resourceSet = domain.getResourceSet();

		// Get the first resource and its root object
		if (!resourceSet.getResources().isEmpty()) {
			val resource = resourceSet.getResources().get(0)
			val app = ComponentAPI.getApplication(resource)
			if (app === null || app.eResource() === null) {
				return null;
			}
			val appUri = app.eResource().getURI().path()
			if (!resource.getContents().isEmpty()) {

				val root = resource.contents.get(0)
				val notifierRoot = DerivedAttributesNotifier.getDerivedAttributesNotifier(appUri, root);
				if (notifierRoot !== null) {
					notifierRoot.analyzeDependencies(root)
					return root
				}
			}
		}
		null
	}

	/**
	 * Ask for a refresh of dependencies (i.e. refreshes all attributes that need to be changes
	 * whenever an external configurator dependent attribute changes
	 */
	static def refreshExternalDependencies(Application app) {
		if(app === null) return;
		refreshApplication(app.eResource)
		DerivedAttributesNotifier.refreshExternalDependencies(app)
	}

	/**
	 * set a configurator enablement state inside an application
	 * @param app the application
	 * @param uri the uri of the configurator
	 * @enabled the state to be associated to the configurator
	 */
	static def setEnabledConfigurator(Application app, String uri, boolean state) {
		val appUri = app.eResource().getURI().path()
		var appConfigurators = applicationConfigurators.get(appUri)
		if (appConfigurators === null) {
			appConfigurators = new BasicEMap<String, Component>()
			applicationConfigurators.put(appUri, appConfigurators)
		}

		if (!state) {
			// remove the configurator from the cache from the state
			appConfigurators.removeKey(uri)
		}
		// Set the state of the configurator to the right state
		adaptEnabledConfigurator(uri, state)

		val appFile = Utils.getFileFromResource(app.eResource().getURI())
		val project = appFile.getProject()
		var Component configurator = null
		val objectname = uri
		val objectUri = URI.createURI(objectname, true)
		configurator = getEObject(app, objectUri.appendFragment("")) as Component
		appConfigurators.put(objectUri.toString, configurator as Component)

		if (configurator !== null) {
			// If various enabled configurators exports the same feature, one must not remove this feature
			// Until we remove the latest configurator exporting it.
			for (exportedfeature : Utils.removeDuplicates(configurator.exportedFeatures)) {
				val name = "/" + project.name + "/" + exportedfeature
				val expuri = URI.createPlatformResourceURI(name, true)
				if (state) {
					logger.debug("    ".repeat(indent) + state + " " + expuri.toString)
					adaptEnabledConfigurator(expuri.toString, true)
				} else {
					var nbEnabled = 0
					for (ref : app.components.filter[isEnabled]) {
						val refname = "/" + project.name + "/" + ref.uri
						val refuri = URI.createPlatformResourceURI(refname, true)
						val refcomp = getConfigurator(app, refuri.toString);
						if (refcomp !== null &&
							Utils.removeDuplicates(refcomp.exportedFeatures).contains(exportedfeature)) {
							nbEnabled++
						}
					}
					if (nbEnabled < 1) {
						// Remove the exported feature from the cache
						logger.debug("    ".repeat(indent) + state + " " + expuri.toString)
						adaptEnabledConfigurator(expuri.toString, false)
						val featurename = "/" + project.name + "/" + exportedfeature
						val featureuri = URI.createPlatformResourceURI(featurename, true)
						adaptEnabledConfigurator(featureuri.toString, false)
					}
				}
			}
		}
	}

	static int indent = 0;

	/**
	 * Refresh the state of enabled Configurators
	 */
	static def refreshEnabledConfigurators(Application app) {
		if(app === null) return;

		synchronized (eInstance) {

			logger.debug("refreshEnabledComponents")
			indent++
			val components = app.components
			val appFile = Utils.getFileFromResource(app.eResource().getURI())
			val project = appFile.getProject()

			// Clear enabledComponent hash for the current project
			for (en : Collections.unmodifiableList(enabledConfigurators.keySet.toList)) {
				val name = "/" + project.name + "/"
				val projectUri = URI.createPlatformResourceURI(name, true)
				if (en.startsWith(projectUri.toString)) {
					enabledConfigurators.remove(en)
				}
			}

			// Now refresh first disabled configurators
			val nonEnabledComponents = components.filter[!isEnabled].toList
			for (ComponentReference c : nonEnabledComponents) {
				val name = "/" + project.getName() + "/" + c.getUri()
				logger.debug("    ".repeat(indent) + "false " + name)
				val objectUri = URI.createPlatformResourceURI(name, true)
				setEnabledConfigurator(app, objectUri.toString, false)
			}

			// now add enabled configurators
			val enComponents = components.filter[isEnabled].toList
			for (ComponentReference c : enComponents) {
				val name = "/" + project.getName() + "/" + c.getUri()
				logger.debug("    ".repeat(indent) + "true " + name)
				val objectUri = URI.createPlatformResourceURI(name, true)
				val comp = ComponentAPI.getEObject(app, objectUri.appendFragment(""));
				if (comp !== null) {
					setEnabledConfigurator(app, objectUri.toString, true)
				}
			}
			indent--
			logger.debug("...done")
		}
	}

	static def resetConfigurator(EObject conf) {
		if(conf === null) return;
		DerivedAttributesNotifier.resetConfigurator(conf)
	}

	/**
	 * Refresh an object, depending its its current editing domain
	 * 
	 * @param obj
	 * @return the original object if we cannot find an editing domain editing it or
	 *         else the new object reference from its editing domain
	 */
	def static EObject refreshComponent(EObject obj) {
		var res = obj;
		if (!isJUnitTest()) {
			val editingDomain = ComponentAPI.getEditingDomain(res);
			if (editingDomain !== null) {
				val objUri = EcoreUtil.getURI(res);
				val newObj = ComponentAPI.getEObjectFromEditingDomain(editingDomain, objUri);
				res = newObj;
			}
		}
		return res;
	}

	/**
	 * Checks whether we are being running unitary tests
	 */
	static def boolean isJUnitTest() {
		for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
			if (element.getClassName().startsWith("org.junit.")) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Get the value of an attribute from its name, using an dedicated API
	 * 
	 * @param destination object the caller object
	 * @param oldValue the value to be used if something went wrong
	 * @param apiObj the API object on which to invoke the method
	 * @param apiMethodName the name of the function to be called on the provided API object in order to get the expected attribute value
	 * @param args the arguments to be passed to the called method (apiMethodName)  
	 */
	static def Object getAttributeValue(EObject destinationObject, Object oldValue, Object apiObj, String apiMethodName,
		Object... args) {
		if (ComponentAPI.isEnabled(destinationObject)) {
			try {
				val clazz = apiObj.class
				for (m : clazz.getMethods()) {
// Look for the good method to be called
					if (m.name.equals(apiMethodName)) {
						// call it with its expected arguments
						val result = m.invoke(apiObj, args);
						if (result !== null) {
							return result
						}
						return oldValue
					}
				}
			} catch (IllegalAccessException e) {
				logger.error(e.message)
				return oldValue
			} catch (InvocationTargetException e) {
				logger.error(e.message)
				return oldValue
			} catch (ClassNotFoundException e) {
				logger.error(e.message)
				return oldValue
			}
		}
	}
	
	/**
	 * Open in editor the resource containing the given component
	 * 
	 * @param retComponent the component to be opened
	 *
	 */
	static def openConfiguratorInEditor(Component retComponent) {
		// LOW: this method should be improved to open the editor only if not already opened
	    if (retComponent === null || retComponent.eResource() === null) return
	    val emfUri = retComponent.eResource().getURI()
	    try {
	    	val wb = PlatformUI.workbench
	        val window = wb.activeWorkbenchWindow
	        if (window === null) return
	        val previousPage = window.activePage
	        val previousPart = previousPage !== null ? previousPage.activePart: null

	        if (emfUri.platformResource) {
	            val platformPath = emfUri.toPlatformString(true)
	            val file = ResourcesPlugin.workspace.root.getFile(new Path(platformPath.substring(1)))
	            val page = window.activePage
	            try {
	                page.openEditor(new FileEditorInput(file),
	                        wb.editorRegistry.getDefaultEditor(file.getFullPath().toString()).getId(),
	                        false)
	            } catch (PartInitException exception) {
	                MessageDialog.openError(window.getShell(), "Open editor fails", exception.getMessage())
	                return
	            }
	        } else {
	            logger.error("Cannot open configurators from non platform resources: " + emfUri)
	        }

	        // Restore focus on previous page/part
	        if (previousPage !== null && previousPart !== null) {
	            try {
	                previousPage.activate(previousPart)
	            } catch (Exception e) {
	                logger.warn("Cannot open reactivate previous part", e)
	            }
	        }
	    } catch (Exception e) {
	        logger.error("Cannot open configurator editor for " + emfUri, e)
	    }
	}
}
