package com.st.stellar.component.impl

import com.st.stellar.component.API.ComponentAPI
import com.st.stellar.component.API.Utils
import com.st.stellar.component.Application
import com.st.stellar.component.Component
import com.st.stellar.component.ComponentPackage
import com.st.stellar.component.ComponentReference
import com.st.stellar.component.ComponentValidation
import com.st.stellar.component.util.ComponentValidator
import java.util.Map
import java.util.regex.Pattern
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.Path
import org.eclipse.emf.common.util.BasicDiagnostic
import org.eclipse.emf.common.util.Diagnostic
import org.eclipse.emf.common.util.DiagnosticChain
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EAttribute
import org.eclipse.emf.ecore.EObject

class MyComponentValidator extends ComponentValidator {

	public static final MyComponentValidator INSTANCE = new MyComponentValidator();

	Application currentApplication = null
	ComponentReference currentComponentRef = null

	new() {
	}

	override protected boolean validate_DataValueConforms(EObject eObject, EAttribute eAttribute,
		DiagnosticChain diagnostics, Map<Object, Object> context) {

		ComponentValidation.validate_DataValueConforms(eObject, eAttribute, diagnostics, context)
	}

	static val ISSUE_CODE_PREFIX = "com.st.stellar.component."

	static val WRONG_URI = ISSUE_CODE_PREFIX + 'wrongUri'
	static val WRONG_NAME = ISSUE_CODE_PREFIX + 'wrongName'
	static val ONE_OF_A_TYPE = ISSUE_CODE_PREFIX + 'oneOfAType'
	static val MISSING_FEATURE = ISSUE_CODE_PREFIX + 'missingfeature'

	override validateApplication(Application application, DiagnosticChain diagnostics, Map<Object, Object> context) {
		currentApplication = application
		super.validateApplication(application, diagnostics, context)
	}

	override validateComponentReference(ComponentReference ref, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		currentApplication = ref.eContainer as Application
		currentComponentRef = ref
		val res = super.validateComponentReference(ref, diagnostics, context)
		
		if( BasicDiagnostic.isInstance(diagnostics) ) {
			val diag = BasicDiagnostic.cast(diagnostics)
			var mess = "";
			if (diag.getSeverity() != Diagnostic.OK) {
				mess = Utils.populateDiagnostics(diag, 99);
			}
			ref.setErrors(mess);
		} else {
			ref.setErrors("");
		}
		res
	}

	override validateComponentReference_checkUri(ComponentReference componentRef, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val app = currentApplication as Application
		val appFile = Utils.getFileFromResource(app.eResource().getURI());
		val project = appFile.getProject();
		var res = true
		if (componentRef.isEnabled) {
			val uriToValidateStr = "/" + project.getName() + "/" + componentRef.uri
			val uriToValidate = URI.createPlatformResourceURI(uriToValidateStr, true);
			val compToValidate = ComponentAPI.getEObject(app, uriToValidate.appendFragment("")) as Component;
			if (compToValidate !== null) {
				val requiredFeatures = Utils.removeDuplicates(compToValidate.requiredFeatures)
				for (feature : requiredFeatures) {
					//var foundFeature = false
					var configuratorsList = newHashSet
					for (ref : app.components.filter[isEnabled]) {
						if (ref !== componentRef) {
							val name = "/" + project.getName() + "/" + ref.uri
							val refUri = URI.createPlatformResourceURI(name, true);
							val comp = ComponentAPI.getEObject(app, refUri.appendFragment("")) as Component;
							if (comp !== null) {
								if (Utils.removeDuplicates(comp.exportedFeatures).contains(feature)) {
									configuratorsList.add(comp)
									//foundFeature = true
								}
							}
						}
					}
					if (configuratorsList.isEmpty) {
						if (diagnostics !== null) {
							diagnostics.add(
								new BasicDiagnostic(Diagnostic.ERROR,
									MyComponentValidator.MISSING_FEATURE,
									99, '''At least one configurator must export feature '«feature»'.''',
									#[componentRef.uri]))
						}
						res = false
					} else if(configuratorsList.size > 1) {
						if (diagnostics !== null) {
							diagnostics.add(
								new BasicDiagnostic(Diagnostic.ERROR,
									MyComponentValidator.MISSING_FEATURE,
									99, '''Only one configurator must export feature '«feature»'.''',
									#[componentRef.uri]))
						}
						res = false
					}
				}
			}
		} else {
			res = true
		}
		return res
	}

	override validateuri_t_checkUri(String nameToValidate, DiagnosticChain diagnostics, Map<Object, Object> context) {
		val app = currentApplication as Application
		if (app === null) {
			return true
		}

		var res = checkUriExists(nameToValidate)
		if (!res) {
			if (diagnostics !== null) {
				diagnostics.add(
					createDiagnostic(Diagnostic.ERROR, WRONG_URI, 99, "_UI_GenericConstraint_diagnostic",
						#["checkUriExists", getValueLabel(ComponentPackage.Literals.URI_T, nameToValidate, context)],
						#[nameToValidate], context))
				res = false
			}
		} else {
			res = checkName(nameToValidate)
			if (!res) {
				if (diagnostics !== null) {
					diagnostics.add(
						createDiagnostic(Diagnostic.ERROR, WRONG_NAME, 99, "_UI_GenericConstraint_diagnostic",
							#["checkName", getValueLabel(ComponentPackage.Literals.URI_T, nameToValidate, context)],
							#[nameToValidate], context))
					res = false
				}
			} else {
				res = checkDuplicates(nameToValidate)
				if (!res) {
					if (diagnostics !== null) {
						diagnostics.add(
							new BasicDiagnostic(Diagnostic.ERROR, ONE_OF_A_TYPE,
								99, '''Only one configurator of type '«nameToValidate.substring(nameToValidate.indexOf('.') + 1)»' must be enabled.''',
								#[nameToValidate])
						)
						res = false
					}
				}
			}
		}
		return res
	}

	override boolean validateuri_t_checkEnable(String uri, DiagnosticChain diagnostics, Map<Object, Object> context) {
		val componentRef = currentComponentRef
		val Application app = currentApplication
		if (componentRef === null) {
			return true
		}

		var res = true
		val uriExtension = componentRef.uri.extension
		var name_list = app.components.filter [
			uri.extension.equals(uriExtension)
		].toList
		// println(uriExtension + ": " + name_list.map[uri].join(",") + " -> " + name_list.size)
		if (name_list.findFirst[isEnabled] === null) {
			if (diagnostics !== null) {
				diagnostics.add(
					new BasicDiagnostic(Diagnostic.ERROR,
						MyComponentValidator.MISSING_FEATURE,
						99, '''At least one configurator of type '«uriExtension»' must be enabled.''',
						#[componentRef.uri]))
				res = false
			}
		}
		// println(componentRef.uri + " -> " + res)
		res
	}

	def boolean checkUriExists(String uriStr) {
		val workspaceRoot = ResourcesPlugin.getWorkspace().getRoot()
		if (currentApplication.eResource !== null) {
			val ifile = Utils.getFileFromResource(currentApplication.eResource.URI)
			val project = ifile.project
			val path = new Path("/" + project.getName() + "/" + uriStr.toString());
			val f = workspaceRoot.getFile(path);
			return f.exists
		}
		return true
	}

	static val REGEX_NAME = Pattern.compile('''^[a-zA-Z]+[a-zA-Z0-9\._]*$''')

	def checkName(String nameToValidate) {

		val parent = currentApplication
		if (parent === null) {
			return true
		}

		val matches = REGEX_NAME.matcher(nameToValidate).matches
		if (!matches) {
			return false
		}
//		// The name is syntactically correct... but may be already present in other component ref
//		// Create the list of configurator names already present inside the application being validated
//		// filter out current component ref (with its old name)
		var name_list = parent.components.filter(it|it !== currentComponentRef).map[it|(it as ComponentReference).uri].
			toList
		if (name_list === null) {
			return true
		}

		// The new name must not be a duplicate
		val noDuplicates = (name_list.filter[it.equals(nameToValidate)].size < 1)
		noDuplicates
	}

	def checkDuplicates(String nameToValidate) {

		if (!currentComponentRef.isEnabled) {
			return true
		}
		val parent = currentApplication
		if (parent === null) {
			return true
		}

		// Create the list of configurator names already present inside the application being validated
		// filter out current component ref (with its old name)
		var name_list = parent.components.filter(it|it !== currentComponentRef && it.isEnabled).map [ it |
			(it as ComponentReference).uri.extension
		].toList
		if (name_list === null) {
			return true
		}

		// The new name must not be a duplicate
		val noDuplicates = (name_list.filter[it.equals(nameToValidate.extension)].size < 1)
		noDuplicates
	}

	def checkNoComponentEnabled(String nameToValidate) {

		if (!currentComponentRef.isEnabled) {
			return true
		}
		val parent = currentApplication
		if (parent === null) {
			return true
		}

//		// Create the list of configurator names already present inside the application being validated
//		// filter out current component ref (with its old name)
		var name_list = parent.components.filter(it|it !== currentComponentRef && it.isEnabled).map [ it |
			(it as ComponentReference).uri.extension
		].toList
		if (name_list === null) {
			return true
		}

		// The new name must not be a duplicate
		val noDuplicates = (name_list.filter[it.equals(nameToValidate.extension)].size < 1)
		noDuplicates
	}

	static def String getExtension(String uri) {
		if (uri === null) {
			return null;
		}

		val lastIndexOfDot = uri.lastIndexOf(".");
		if (lastIndexOfDot == -1) {
			return null; // No extension found
		}

		return uri.substring(lastIndexOfDot + 1);
	}

}
