package com.st.stellar

import java.lang.reflect.InvocationTargetException
import java.util.Collections
import java.util.Iterator
import java.util.Stack
import org.apache.log4j.LogManager
import org.apache.log4j.Logger
import org.eclipse.core.runtime.Platform
import org.eclipse.emf.ecore.EClass
import org.eclipse.emf.ecore.EFactory
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EReference
import org.osgi.framework.Bundle
import java.util.stream.Collectors

class StellarHelper {
	static Logger logger = LogManager.getLogger(StellarHelper)

	/** 
	 * Create references that are needed from one parent object
	 * 
	 * For all required references we accumulate the reference objects to be created
	 * using a stack of items to be processed
	 * 
	 * @param factory the factory to be used to create the instances
	 * @param parent the parent object for which the references must be created
	 */
	def static void createRequiredReferences(EFactory factory, EObject parent) {

		var stack = new Stack<EObject>();

		// Add first object to the stack
		stack.push(parent)

		// Traverse the stack one element at a time while the stack is not empty.
		// When processing one element we may add some new elements to be processed at a later stage
		while (!stack.isEmpty) {
			// Use the stack as a FIFO
			val current = stack.pop

			var references = current.eClass().getEAllStructuralFeatures().filter(EReference)
			for (var Iterator<EReference> iterator = references.iterator(); iterator.hasNext();) {
				var ref = iterator.next()

				if (ref.lowerBound === 1 && ref.upperBound === 1) {
					var EClass type = ref.getEType() as EClass

					// Create the expected referenced object and assign it to the parent
					val eFactory = type.getEPackage().getEFactoryInstance();
					var EObject refInstance = eFactory.create(type)
					current.eSet(ref, refInstance)

					// Push the new instance to be processed
					stack.push(refInstance)
				} else if (ref.lowerBound > 0) {
					if (ref.upperBound == -1) {
						var EClass type = ref.getEType() as EClass
						val previous = current.eContents.filter(type.class)
						if (previous.size < 1) {
							// Create the expected referenced object and assign it to the parent
							val eFactory = type.getEPackage().getEFactoryInstance();

							var EObject refInstance = eFactory.create(type)
							current.eSet(ref, Collections.singleton(refInstance));

							// Push the new instance to be processed
							stack.push(refInstance)
						}
					} else if (ref.upperBound > 0) {
						var EClass type = ref.getEType() as EClass
						var previous = current.eContents.filter(type.class)
						val eFactory = type.getEPackage().getEFactoryInstance();
						var nb = previous.size
						val array = newArrayList
						// Create the expected instances...
						while (nb < ref.upperBound) {
							var EObject refInstance = eFactory.create(type)
							// Push the new instance to be processed
							stack.push(refInstance)
							array.add(refInstance)
							nb++
						}
						//  ...and assign them to the parent
						current.eSet(ref, array.stream.collect(Collectors.toList))
					}
				}
			}
		}
	}

	/**
	 * Search for all extension defining generated packages and get the plugin which
	 * declares the given component
	 * 
	 * @param component the component for which we want to find the contributor
	 *                  plugin
	 * @return
	 */
	static def Bundle getBundle(EObject object) {

		val extensionPoint = Platform.getExtensionRegistry().getExtensionPoint(
			"org.eclipse.emf.ecore.generated_package") // $NON-NLS-1$
		val resourceURI = object.eResource().getURI()
		for (ext : extensionPoint.getExtensions()) {
			for (configurationElement : ext.getConfigurationElements()) {
				if (configurationElement.getAttribute("uri").equals(resourceURI.toString())) { // $NON-NLS-1$
					return Platform.getBundle(configurationElement.getContributor().getName())
				}
			}
		}
		return null
	}

	def static addDependencies(Object parent) {
		val op = parent.class.methods.findFirst[name.equals("addDependencies")]
		if (op !== null) {
			try {
				op.invoke(parent)
			} catch (InvocationTargetException e) {
				logger.error("Cannot invoke method 'addDependencies'")
				logger.error(e.message)
			}
		}

	}

	def static String getItems(
		Object parent,
		String getItemsFunctionName
	) {
		val op = parent.class.methods.findFirst[name.equals(getItemsFunctionName)]
		if (op !== null) {
			try {
				val res = op.invoke(parent).toString
				res
			} catch (NullPointerException e) {
				"Undefined"
			} catch (InvocationTargetException e) {
				logger.error("Cannot invoke method '" + getItemsFunctionName + "'")
				logger.error(e.message)
				"Undefined"
			}
		}

	}
}
