package com.st.stellar.pinmap.gpiocfg.codeGenerator.runtime

import com.google.inject.Inject
import com.google.inject.Provider
import com.st.stellar.pinmap.gpio.gpioDsl.ConfigurablePin
import com.st.stellar.pinmap.gpioCfg.Module
import com.st.stellar.pinmap.gpioCfg.PinConfiguration
import com.st.stellar.pinmap.gpioCfg.PinSetting
import com.st.stellar.pinmap.gpioCfg.PinSettings
import com.st.stellar.pinmap.helpers.GpioCfgHelper
import java.util.Locale
import java.util.regex.Pattern
import org.eclipse.core.resources.IProject
import org.eclipse.core.resources.IResource
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.CoreException
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.core.runtime.Platform
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.resource.ResourceSet
import org.eclipse.xtext.generator.AbstractGenerator
import org.eclipse.xtext.generator.IFileSystemAccess
import org.eclipse.xtext.generator.IFileSystemAccess2
import org.eclipse.xtext.generator.IGeneratorContext
import org.eclipse.xtext.generator.JavaIoFileSystemAccess
import org.eclipse.xtext.generator.OutputConfiguration
import org.spiritconsorsium.xml.spiritmodel.Spirit.FieldType
import org.spiritconsorsium.xml.spiritmodel.Spirit.RegisterType

// Implements SDD_GpioCfg_CODE_GENERATION
class GpioCfgCodeGenerator extends AbstractGenerator {

	@Inject Provider<ResourceSet> rsp
	@Inject extension GpioCfgHelper

	OutputConfiguration outputCfg

	static int pageWidth = 50;

	override void doGenerate(Resource resource, IFileSystemAccess2 fsa, IGeneratorContext context) {
		val resourceSet = rsp.get
		val r = resourceSet.getResource(resource.URI, true)
		if(r.contents.empty) return;

		try {
			val head = r.contents.head
			val cfg = head as PinConfiguration
			if (cfg.pinSetting instanceof PinSettings) {
				val pinSettings = cfg.pinSetting

				var outputFolder = ""
				if (pinSettings.outputFolder !== null) {
					outputFolder = "../" + pinSettings.outputFolder + "/"
				} else {
					if (fsa instanceof JavaIoFileSystemAccess) {
						// We are generating from headless generator
						val fsa2 = fsa as JavaIoFileSystemAccess
						outputCfg = fsa2.outputConfigurations.get("GpioCfgCode")

						outputFolder = outputCfg.getOutputDirectory() + "/"
						outputFolder = '../' + outputFolder
					} else {
						outputFolder = "./src-gen/"
						outputFolder = '../' + outputFolder
					}
				}

				val bootheader = generatePinSettings(pinSettings)
				// fsa.generateFile(outputFolder + pinSettings.name + ".c", bootheader)
				fsa.generateFile(outputFolder + pinSettings.name + ".h", IFileSystemAccess.DEFAULT_OUTPUT, bootheader)
			}
			updateExplorer(cfg)
		} catch (Exception e) {
			e.printStackTrace(System.err)
		}
	}

	static String PINMAPCFG_BUNDLE_ID = "com.st.stellar.pinmap.gpiocfg"

	static def String getVersion() {
		if (Platform.running) {
			val ideBundle = Platform.getBundle(PINMAPCFG_BUNDLE_ID)
			ideBundle.getVersion().toString
		} else {
			GpioCfgHelper.GPIOCFG_VERSION
		}
	}

	static def String getPluginName() {
		if (Platform.running) {
			val ideBundle = Platform.getBundle(PINMAPCFG_BUNDLE_ID)
			ideBundle.getSymbolicName()
		} else {
			PINMAPCFG_BUNDLE_ID
		}
	}

	def generatePinSettings(PinSettings pinSettings) {
		var fieldStr = '''
			/****************************************************************************
			*
			* Copyright (c) 2022-2026 STMicroelectronics - All Rights Reserved
			*
			* License terms: STMicroelectronics Proprietary in accordance with licensing
			* terms SLA0098 at www.st.com.
			*
			* THIS SOFTWARE IS DISTRIBUTED "AS IS," AND ALL WARRANTIES ARE DISCLAIMED,
			* INCLUDING MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
			*
			* PLEASE NOTE THAT THIS FILE IS GENERATED FROM A CONFIGURATION FILE,
			* USING «pluginName», Version «version.toString»
			* MODIFYING IT IMPLIES POTENTIAL LOSS OF YOUR MODIFICATIONS.
			*
			*****************************************************************************/
			/**
			 * @file    «pinSettings.name».h
			 * @brief   SR5E1-«GpioCfgHelper.getBoard(pinSettings).name.toUpperCase(Locale.US)» board header file.
			 *
			 * @addtogroup PLATFORM
			 * @addtogroup BOARD
			 * @ingroup PLATFORM
			 * @addtogroup «pinSettings.name.toUpperCase(Locale.US)»
			 * @ingroup BOARD
			 * @{
			 */
			
			#ifndef _«pinSettings.name.toUpperCase(Locale.US)»_H_
			#define _«pinSettings.name.toUpperCase(Locale.US)»_H_
			
			#include <gpio.h>
			#include <platform.h>
			#include <typedefs.h>
			
			«pinSettings.familyDescriptionComment»
			
			/*===========================================================================*/
			/* Module constants.                                                         */
			/*===========================================================================*/
			
			#define «GpioCfgHelper.getBoard(pinSettings).name.toUpperCase(Locale.US)»	1
			
			«FOR mod : GpioCfgHelper.getAllModules(pinSettings)»
				«IF mod.comment === null»
					/*--------------------- «mod.name.name» pins definitions ---------------------*/
				«ELSE»
					/*
					 * «mod.comment»
					 */
				«ENDIF»
				«FOR pin : mod.pins»
					«val spaces = pageWidth - pin.toUpper.length - "#define ".length - 1»
					#define «pin.toUpper» gpio_iopack(«pin.portName», «pin.pad») /* «pin.getPadName» */
					«pin.getIODefinitions»
					
				«ENDFOR»
				
				«ENDFOR»
			
			/*===========================================================================*/
			/* Module pre-compile time settings.                                         */
			/*===========================================================================*/
			
			/*===========================================================================*/
			/* Derived constants and error checks.                                       */
			/*===========================================================================*/
			
			/*===========================================================================*/
			/* Module data structures and types.                                         */
			/*===========================================================================*/
			
			/*===========================================================================*/
			/* Module macros.                                                            */
			/*===========================================================================*/
			
			/*===========================================================================*/
			/* External declarations.                                                    */
			/*===========================================================================*/
			
			#if !defined(_FROM_ASM_)
			#ifdef __cplusplus
			extern "C" {
			#endif
			
			#ifdef __cplusplus
			}
			#endif
			#endif /* _FROM_ASM_ */
			
			/*===========================================================================*/
			/* Module inline functions.                                                  */
			/*===========================================================================*/
			
			
			__STATIC_INLINE void «pinSettings.name»_init(void) {
				
			«FOR mod : GpioCfgHelper.getAllModules(pinSettings)»
				«mod.gpio_set_pin_modes»
				
			«ENDFOR»
			}
			
			/*===========================================================================*/
			/* Module late inclusions.                                                   */
			/*===========================================================================*/
			
			#endif /* _«pinSettings.name.toUpperCase(Locale.US)»_H_ */
			
			/** @} */
			
		'''
		fieldStr
	}

	def String gpio_set_pin_modes(Module mod) {
		var res = ""

		var commentDisplayed = false
		val comment = '''	/* «mod.name.name» */
		'''

		for (pin : mod.pins) {
			if (pin.ioFields !== null) {
				if (!commentDisplayed) {
					commentDisplayed = true
					res += comment
				}
				val ioMode = '''	gpio_set_pin_mode(«pin.toUpper», «pin.toUpper»_CFG);
				'''
				res += '''«ioMode»'''
			}
		}
		res
	}

	def String toUpper(PinSetting ps) {
		ps.name.toUpperCase(Locale.US)
	}

	def String getIODefinitions(PinSetting ps) {
		var ioDef = ""
		var instances = newArrayList
		// for (reg : ps.config.ref) {
		if (ps.ioFields === null) {
			return '''/* No IO definition for «ps.toUpper» */'''
		}
		val regInstances = ps.ioFields.reg
		for (inst : regInstances) {
			val field = inst.attr as FieldType
			val reg = field.eContainer as RegisterType
			if (reg.isSupported) {
				instances.add(inst)
			}
		}

		if (instances.size > 0) {
			val spaces = Math.max(0, pageWidth - ps.toUpper.length - "#define ".length - 2)
			var i = 1

			var lines = newArrayList
			if (instances.size > 0) {
				lines.add('''
				#define «ps.toUpper»_CFG«" ".repeat(spaces)»\''')
			}
			var line = ""
			if (instances.size == 1) {
				line = "\t(" + instances.get(0).toIO + ")"
				line += " ".repeat(Math.max(pageWidth - line.length - 1, 1))
				lines.add(line)
			} else {
				for (inst : instances) {
					if (i == 1) {
						line = "\t(" + inst.toIO + " |"
						line += " ".repeat(Math.max(pageWidth - line.length - 1, 1))
						line += "\\"
					} else if (i == instances.size) {
						line = "\t " + inst.toIO + ")"
					} else {
						line = "\t " + inst.toIO + " |"
						line += " ".repeat(Math.max(pageWidth - line.length - 1, 1))
						line += "\\"
					}
					lines.add(line)
					i++
				}
			}
			ioDef = '''«FOR l : lines»
				«l»
			«ENDFOR»'''
		} else {
			ioDef = '''/* No register instances for «ps.toUpper» */'''
		}
		ioDef
	}

	def getPadName(PinSetting setting) {
		val configurations = setting.pin.pinConfigurations
		val first = configurations.filter(ConfigurablePin).get(0);
		var pattern = Pattern.compile("P([^\\d]+)(\\d+)")
		var matcher = pattern.matcher(first.ref.name)
		if (matcher.find()) {
			return "P" + matcher.group(1) + "[" + matcher.group(2) + "]"
		}
		return first.ref.label
	}

	def getPad(PinSetting setting) { // SIUL2_PAD_0
		val configurations = setting.pin.pinConfigurations
		val first = configurations.filter(ConfigurablePin).get(0);
		var pattern = Pattern.compile("P([\\d]+)_(\\d+)")
		var matcher = pattern.matcher(first.ref.name)
		if (matcher.find()) {
			return "GPIO_PIN_" + matcher.group(2)
		}
		return first.ref.name
	}

	val ports = #["GPIO_PORT_A", "GPIO_PORT_B", "GPIO_PORT_C", "GPIO_PORT_D", "GPIO_PORT_E", "GPIO_PORT_F",
		"GPIO_PORT_G", "GPIO_PORT_H", "GPIO_PORT_I", "GPIO_PORT_J", "GPIO_PORT_K", "GPIO_PORT_L", "GPIO_PORT_M",
		"GPIO_PORT_N", "GPIO_PORT_Q", "GPIO_PORT_R", "GPIO_PORT_S", "GPIO_PORT_T", "GPIO_PORT_V", "GPIO_PORT_W",
		"GPIO_PORT_X", "GPIO_PORT_Y"]

	def getPortName(PinSetting setting) { // SIUL2_PORT_D
		val configurations = setting.pin.pinConfigurations
		val first = configurations.filter(ConfigurablePin).get(0); // TODO: Modify this part for multiple boundings
		var pattern = Pattern.compile("P([\\d]+)_(\\d+)")
		var matcher = pattern.matcher(first.ref.name)
		if (matcher.find()) {
			val index = Integer.parseInt(matcher.group(1))
			return ports.get(index)
		}
		return first.ref.name
	}

	def String getFamilyDescriptionComment(PinSettings ps) {
		var StringBuffer sb = new StringBuffer
		val desc = ps.familyDescription
		if (desc.length > 0) {
			sb.append("/*===========================================================================*/\n")
			for (line : desc.split('\n')) {
				val l = line.trim.replaceAll("\n", "")
				sb.append("/* " + l + (l.length > 73 ? "" : " ".repeat(73 - l.length)) + " */\n")
			}
			sb.append("/*===========================================================================*/\n")
		}
		sb.toString
	}

	static def IProject getProject(EObject model) {
		var IProject project = null
		val uri = model.eResource().getURI()
		if (!uri.isFile()) {
			project = null;
		}
		val projectName = uri.segmentsList().get(1).toString();
		project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		return project;
	}

	static def void updateExplorer(EObject model) {

		val project = getProject(model);
		if (project !== null) {
			try {
				project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor())
			} catch (CoreException e) {
				// TODO Auto-generated catch block
				e.printStackTrace()
			}
		}
	}

}
