package com.st.stellar.pinmap.gpiocfg.excelGenerator.runtime

import com.google.inject.Inject
import com.google.inject.Provider
import com.st.stellar.pinmap.gpio.gpioDsl.ConfigurablePin
import com.st.stellar.pinmap.gpio.gpioDsl.Family
import com.st.stellar.pinmap.gpio.gpioDsl.NotConnectedPin
import com.st.stellar.pinmap.gpio.gpioDsl.Pin
import com.st.stellar.pinmap.gpio.gpioDsl.PinmapModel
import com.st.stellar.pinmap.gpio.gpioDsl.SpecialPin
import com.st.stellar.pinmap.gpio.gpioDsl.SystemFunctionPin
import com.st.stellar.pinmap.gpio.helpers.ResourceUtils
import com.st.stellar.pinmap.gpioCfg.Include
import com.st.stellar.pinmap.gpioCfg.PinConfiguration
import com.st.stellar.pinmap.gpioCfg.PinSettings
import com.st.stellar.pinmap.helpers.GpioCfgHelper
import java.io.Closeable
import java.io.File
import java.io.FileNotFoundException
import java.io.FileOutputStream
import java.io.IOException
import java.util.ArrayList
import java.util.List
import org.apache.poi.hssf.usermodel.HSSFWorkbook
import org.apache.poi.ss.usermodel.IndexedColors
import org.eclipse.core.resources.IProject
import org.eclipse.core.resources.IResource
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.CoreException
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.core.runtime.Path
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.resource.ResourceSet
import org.eclipse.xtext.generator.AbstractGenerator
import org.eclipse.xtext.generator.IFileSystemAccess2
import org.eclipse.xtext.generator.IGeneratorContext
import org.spiritconsorsium.xml.spiritmodel.Spirit.DocumentRoot
import org.spiritconsorsium.xml.spiritmodel.Spirit.MemoryMapType
import org.spiritconsorsium.xml.spiritmodel.Spirit.RegisterType

class ExcelGenerator extends AbstractGenerator {

	@Inject extension GpioCfgHelper
	@Inject Provider<ResourceSet> rsp

	override void doGenerate(Resource r, IFileSystemAccess2 fsa, IGeneratorContext context) {

		val resourceSet = rsp.get
		val resource = resourceSet.getResource(r.URI, true)
		// Apply Excel generation
		if(resource.contents.head === null) return
		val configuration = resource.contents.head as PinConfiguration

		// Ugly code to check whether we need to generate HTML code or not
		// TODO: find a more elegant code
		val boolean generateExcel = context.cancelIndicator.class.name.contains("GenerateExcelFileCancelIndicator")

		if (generateExcel) {

			val pinSettings = configuration.pinSetting
			if(pinSettings === null) return;

			if(GpioCfgHelper.getBoard(pinSettings.includedBoard).package === null) return
			var n = (GpioCfgHelper.getBoard(pinSettings.includedBoard).package.eContainer as Family).name
			n = n.replaceAll('x', pinSettings.name)

			// Implements SDD_GpioCfg_EXPORT_EXCEL
			// Excel generation
			// var fileName = "../data-gen/" + n + ".xls"
			val fileName = pinSettings.getGenerationFileName("xls")
			// val fileName = file.toString.replace('L/', '../data-gen/')
			// fsa.generateFile("../data-gen/" + n + ".txt", "")
			// fsa.deleteFile("../data-gen/" + n + ".txt")
			var excelFileUri = fsa.getURI(fileName, "GpioCfgCode")
			val file2 = ResourcesPlugin.workspace.root.getFile(new Path(excelFileUri.toPlatformString(true)))
			generateExcel(file2.rawLocation.toOSString, pinSettings)

			updateExplorer(configuration);
		}
	}

	def static toText(Include imp) {
		val PinmapModel importedPinmap = ResourceUtils.resourceToEObject(ResourceUtils.openImport(imp.eResource,
			imp.importURI)) as PinmapModel;
		return '''
			Pinmap model file: «imp.importURI»:
			«IF importedPinmap === null»
				File cannot be opened.
			«ELSE»
				Number of families: «importedPinmap.families.size»
				«FOR imp2 : importedPinmap.includes»
					«toText(imp2)»
				«ENDFOR»
			«ENDIF»
		''';
	}

	static def List<RegisterType> registers(DocumentRoot root) {
		var registerTypes = new ArrayList<RegisterType>()
		var a = root.component
		var memoryMaps = a.memoryMaps
		for (m : memoryMaps.eContents.filter(MemoryMapType)) {
			var mmap = m as MemoryMapType
			var listAB = mmap.addressBlock

			for (ab : listAB) {
				registerTypes.addAll(ab.register)
			}
		}
		registerTypes
	}

	def static toText(com.st.stellar.pinmap.gpio.gpioDsl.Include imp) {
		val DocumentRoot importedRegister = ResourceUtils.resourceToEObject(ResourceUtils.openImport(imp.eResource,
			imp.importURI)) as DocumentRoot;
		return '''
			Register model file: «imp.importURI»:
			«IF importedRegister === null»
				File cannot be opened.
			«ELSE»
				Number of registers: «importedRegister.registers.size»
			«ENDIF»
		''';
	}

	val String[] columns = #["#", "Pin identifier", "Active", "Function direction", "FunctionName", "Peripheral",
		"Function alternate", "Pin advanced settings"]

	def generateExcel(
		String filename,
		PinSettings settings
	) {
		_gpioCfgHelper = GpioCfgHelper.instance
		var workbook = new HSSFWorkbook()

		/* CreationHelper helps us create instances of various things like DataFormat, 
		 Hyperlink, RichTextString etc, in a format (HSSF, XSSF) independent way */
		var createHelper = workbook.creationHelper

		// Create a Sheet
		var sheet = workbook.createSheet("PinConfiguration")

		// Create a Font for styling header cells
		val headerFont = workbook.createFont()
		headerFont.setFontHeightInPoints(Short.parseShort("14"));
		headerFont.setColor(IndexedColors.RED.getIndex());

		// Create a CellStyle with the font
		var headerCellStyle = workbook.createCellStyle();
		headerCellStyle.setFont(headerFont);

		// Create a Row
		val headerRow = sheet.createRow(0);

		// Create cells
		for (var i = 0; i < columns.length; i++) {
			val cell = headerRow.createCell(i);
			cell.setCellValue(columns.get(i));
			cell.setCellStyle(headerCellStyle);
		}

		// Create Cell Style for formatting Date
		val dateCellStyle = workbook.createCellStyle();
		dateCellStyle.setDataFormat(createHelper.createDataFormat().getFormat("dd-MM-yyyy"));

		// Create the entries in the sheet
		var rowNb = 1
		for (mod : GpioCfgHelper.getAllModules(settings))
			for(pin : mod.pins) {
				val row = sheet.createRow(rowNb++);
				for (var i = 0; i < columns.length; i++) {
					// String text = provider.getColumnText(pin, i);
					// Label data = new Label(i, row, text, formatData);
					// sheet.addCell(data);
					row.createCell(0).setCellValue(pin.pin.name);
					row.createCell(1).setCellValue(pin.name);
					row.createCell(2).setCellValue("true");
					row.createCell(3).setCellValue(pin.config.direction.getName);
					row.createCell(4).setCellValue(GpioCfgHelper.getFunctionName(pin));

					val peripheral = pin.associatedPeripheral
					row.createCell(5).setCellValue(peripheral);
					row.createCell(6).cellValue = GpioCfgHelper.getFunctionName(pin)
					row.createCell(7).cellValue = pin.getIODefinitions.replaceAll("GPIO_MODE_", "")
				}
			}

		for (Pin pin : GpioCfgHelper.getBoard(settings.includedBoard).package.pins) {
			if (!(pin.pinConfigurations.get(0) instanceof NotConnectedPin)) {
				val row = sheet.createRow(rowNb++);
				for (var i = 0; i < columns.length; i++) {
					// String text = provider.getColumnText(pin, i);
					// Label data = new Label(i, row, text, formatData);
					// sheet.addCell(data);
					row.createCell(0).setCellValue(pin.name)
					row.createCell(2).setCellValue("NONE")
					row.createCell(3).setCellValue(direction(pin.pinConfigurations.get(0)))
					row.createCell(4).setCellValue(valueOfCell(pin.pinConfigurations.get(0)))
//				row.createCell(4).setCellValue(pin.config.alt.name);
				}
			}
		}

// Resize all columns to fit the content size
		for (var i = 0; i < columns.length; i++) {
			sheet.autoSizeColumn(i);
		}

		// Write the output to a file, then read it back to get its CharSequence
		try {
			val fileOut = new FileOutputStream(new File(filename))
			workbook.write(fileOut)
			fileOut.close

		} catch (FileNotFoundException e) {
			println(e.message)
		}
	}

	dispatch def direction(ConfigurablePin config) {
		""
	}

	dispatch def direction(SystemFunctionPin config) {
		config.direction.getName
	}

	dispatch def direction(SpecialPin config) {
		config.direction.getName
	}

	dispatch def direction(NotConnectedPin config) {
		"NotConnectedPin"
	}

	dispatch def valueOfCell(ConfigurablePin config) {
		config.ref.name
	}

	dispatch def valueOfCell(SystemFunctionPin config) {
		config.ref.name
	}

	dispatch def valueOfCell(SpecialPin config) {
		config.ref.name
	}

	dispatch def valueOfCell(NotConnectedPin config) {
		"NotConnectedPin"
	}

	def void closeStream(Closeable stream) {
		if (stream !== null) {
			try {
				stream.close();
			} catch (IOException e) {
				// Ignore
			}
		}
	}

	static def IProject getProject(EObject model) {
		var IProject project = null
		val uri = model.eResource().getURI()
		if (!uri.isFile()) {
			project = null;
		}
		val projectName = uri.segmentsList().get(1).toString();
		project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		return project;
	}

	static def void updateExplorer(EObject model) {

		val project = getProject(model);
		if (project !== null) {
			try {
				project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor())
			} catch (CoreException e) {
				// TODO Auto-generated catch block
				e.printStackTrace()
			}
		}
	}

}
