package com.st.stellar.pinmap.gpiocfg.glsp.server

import com.google.inject.Inject
import com.st.stellar.pinmap.gpio.gpioDsl.ConfigurablePin
import com.st.stellar.pinmap.gpio.gpioDsl.Function
import com.st.stellar.pinmap.gpio.gpioDsl.FunctionRef
import com.st.stellar.pinmap.gpio.gpioDsl.Pin
import com.st.stellar.pinmap.gpio.gpioDsl.SpecialPin
import com.st.stellar.pinmap.gpio.gpioDsl.SystemFunctionPin
import com.st.stellar.pinmap.gpioCfg.PinSetting
import com.st.stellar.pinmap.gpiocfg.glsp.server.model.GpioCfgModelState
import com.st.stellar.pinmap.helpers.GpioCfgHelper
import java.util.HashMap
import java.util.LinkedHashMap
import java.util.List
import java.util.Map
import java.util.Optional
import org.eclipse.emf.ecore.EObject
import org.eclipse.glsp.graph.GHtmlRoot
import org.eclipse.glsp.graph.GModelElement
import org.eclipse.glsp.graph.GraphFactory
import org.eclipse.glsp.server.features.popup.PopupModelFactory
import org.eclipse.glsp.server.features.popup.RequestPopupModelAction

import static org.eclipse.glsp.graph.util.GraphUtil.bounds

class GpioCfgPopupModelFactory implements PopupModelFactory {

	@Inject
	protected GpioCfgModelState modelState;

	def Pin lookupPin(GModelElement element) {
		val selectedPinConfigs = modelState.index.getEObject(element.id)
		val ps = selectedPinConfigs.stream().filter(it|Pin.isInstance(it)).map(it|Pin.cast(it)).findFirst();
		return ps.orElse(null);
	}

	def PinSetting lookupPinSetting(GModelElement element) {
		val selectedPinConfigs = modelState.index.getEObject(element.id)
		val selectedPin = selectedPinConfigs.stream().filter(it|Pin.isInstance(it)).map(it|Pin.cast(it)).findFirst();
		if (selectedPin.isPresent) {
			val config = modelState.pinConfiguration
			val ps = GpioCfgHelper.getAllPinSettings(config).findFirst[it|it.pin.name.equals(selectedPin.get.name)]
			return ps;
		}
		return null
	}

	def String generateBody(Pin pin, PinSetting ps) {
		if (pin === null) {
			return null
		}

		var psPeriph = ""
		var psFct = ""
		if (ps !== null) {
			val fct = ps.config.alt
			if (fct !== null) {
				psFct = fct.name
				psPeriph = fct.peripheralRef.label
			}
		}

		val pinMap = new HashMap<Pin, HashMap<String, String>>

		var tooltipMap = pinMap.get(pin)
		if (tooltipMap === null) {
			pinMap.put(pin, new HashMap<String, String>)
			tooltipMap = pinMap.get(pin)
		}

		var tooltips = ""
		for (cfg : pin.pinConfigurations) {
			var tooltip = ""
			var periph = ""
			if (cfg instanceof SystemFunctionPin) {
				val sfp = cfg as SystemFunctionPin
				periph = "System"

				tooltip = tooltipMap.get(periph)
				if (tooltip === null) {
					tooltipMap.put(periph, "")
					tooltip = tooltipMap.get(periph)
				}
				tooltip += sfp.ref.label + " (" + sfp.direction.getName + ")" + NL
				tooltipMap.put(periph, tooltip)
			} else if (cfg instanceof SpecialPin) {
				val sp = cfg as SpecialPin
				periph = "Special"
				tooltip = tooltipMap.get(periph)
				if (tooltip === null) {
					tooltipMap.put(periph, "")
					tooltip = tooltipMap.get(periph)
				}
				tooltip += sp.ref.label + " (" + sp.direction.getName + ")"
				tooltip += NL
				tooltipMap.put(periph, tooltip)
			} else if (cfg instanceof ConfigurablePin) {
				val sfp = cfg as ConfigurablePin
				if (tooltips.length < 1) {
					tooltips = "<b>" + pin.name + ": " + sfp.ref.label + "</b><br/><pre>"
				}
				val functionName = psFct
				val configuredFunction = sfp.ref.functions.stream.filter [ it |
					it.name.name.equals(functionName)
				].findFirst

				val alternateFunctions = new LinkedHashMap<String, List<Function>>();
				for (FunctionRef daf : cfg.getRef().getFunctions()) {
					if (daf.getAlt() !== null) {
						// println(daf.alt)
						var af = alternateFunctions.get(daf.alt)
						if (af === null) {
							af = newArrayList
							alternateFunctions.put(daf.alt, af)
						}
						// println(daf.alt+ " added ")
						af.add(daf.getName());
					// println(af.map(it|it.name).join(","))
					}
				}

				if (configuredFunction.isPresent) {
					val configuredFunctionName = configuredFunction.get.name.name
					val directAnalogFunctions = sfp.ref.functions.filter [ it |
						isSelectedFunction(alternateFunctions, it, configuredFunction.get)
					]
					for (c : sfp.ref.functions) {
						val fctName = c.name.name
						periph = c.name.peripheralRef.label
						var t = tooltipMap.get(periph)
						if (t === null) {
							tooltipMap.put(periph, "")
							t = tooltipMap.get(periph)
						}
						if (directAnalogFunctions.contains(c)) {
							t += "&#160;&#160;&#160;<b>(" + c.direction + ") " + fctName + "</b><br/>"
						} else {
							if (configuredFunctionName.equals(fctName)) {
								t += "&#160;&#160;&#160;<b>(" + c.direction + ") " + fctName + "</b><br/>"
							} else {
								t += "&#160;&#160;&#160;(" + c.direction + ") " + fctName + NL
							}
						}
						tooltipMap.put(periph, t)
					}
				} else {
					for (c : sfp.ref.functions) {
						periph = c.name.peripheralRef.label
						var t = tooltipMap.get(periph)
						if (t === null) {
							tooltipMap.put(periph, "")
							t = tooltipMap.get(periph)
						}
						val fctName = c.name.name
						t += "&#160;&#160;&#160;(" + c.direction + ") " + fctName + NL
						tooltipMap.put(periph, t)
					}
				}
				tooltip += tooltipMap.get(periph)
				tooltips += "</pre>"

			} else {
				tooltip += "&#160;&#160;&#160;" + cfg.toString + NL
			}

		}

		for (key : tooltipMap.keySet) {
			if (key.equals(psPeriph)) {
				tooltips += "<b>" + key + "</b><br/>"
			} else {
				tooltips += "<em>" + key + "</em><br/>"
			}
			tooltips += tooltipMap.get(key) // + "<br/>"
		}
		tooltips
	}

	protected def boolean isSelectedFunction(Map<String, List<Function>> functions, FunctionRef configuredFunction,
		FunctionRef searchedFunction) {
		if (configuredFunction.alt === null) {
			return false;
		}
		if (searchedFunction.alt === null) {
			return false;
		}
//		println("Checking " + searchedFunction.name.name + " with " + configuredFunction.name.name)
//		println(functions.get(searchedFunction.alt).map(it|it.name).join(","))
		return functions.get(searchedFunction.alt).map(it|it.name).contains(configuredFunction.name.name)
//		if (a) {
//			println(configuredFunction.name.name + " found")
//		}
//		return a
	}

	dispatch def String generateTitle(PinSetting ps) {
		return null
	}

	dispatch def String generateTitle(EObject task) {
		return null
	}

	static final String NL = "<br/>";

	override Optional<GHtmlRoot> createPopupModel(GModelElement element, RequestPopupModelAction action) {

		if (element !== null) {
			if (!modelState.popupState) {

				val pinSetting = lookupPinSetting(element);
				if (pinSetting !== null) {
					val root = GraphFactory.eINSTANCE.createGHtmlRoot
					val bounds = action.bounds
					root.setCanvasBounds(bounds(bounds.x, bounds.y, bounds.width, bounds.height))
					root.type = "html"
					root.id = "sprotty-popup"
					val title = ""
					val p1 = GraphFactory.eINSTANCE.createGPreRenderedElement
					p1.type = "pre-rendered"
					p1.id = "popup-title"
					p1.code = '''<div class="sprotty-popup-title">«title»</div>'''
					root.children.add(p1)

					val body = generateBody(pinSetting.getPin(), pinSetting)
					if (body !== null) {
						val p2 = GraphFactory.eINSTANCE.createGPreRenderedElement
						p2.type = "pre-rendered"
						p2.id = "popup-body"
						p2.code = '''<div class="sprotty-popup-body">«body»</div>'''
						root.children.add(p2)
					}
					return Optional.of(root)
				} else {
					val ps = lookupPin(element);
					if (ps !== null) {
						val root = GraphFactory.eINSTANCE.createGHtmlRoot
						val bounds = action.bounds
						root.setCanvasBounds(bounds(bounds.x, bounds.y, bounds.width, bounds.height))
						root.type = "html"
						root.id = "sprotty-popup"
						val title = ""
						val p1 = GraphFactory.eINSTANCE.createGPreRenderedElement
						p1.type = "pre-rendered"
						p1.id = "popup-title"
						p1.code = '''<div class="sprotty-popup-title">«title»</div>'''
						root.children.add(p1)

						val body = generateBody(ps, null)
						if (body !== null) {
							val p2 = GraphFactory.eINSTANCE.createGPreRenderedElement
							p2.type = "pre-rendered"
							p2.id = "popup-body"
							p2.code = '''<div class="sprotty-popup-body">«body»</div>'''
							root.children.add(p2)
						}
						return Optional.of(root)
					}
				}
			}
		}
		return Optional.empty
	}
}
