package com.st.stellar.pinmap.gpiocfg.glsp.server.model

import com.google.common.collect.HashMultimap
import com.google.common.collect.Multimap
import com.google.inject.Inject
import com.st.stellar.pinmap.gpio.gpioDsl.ConfigurablePin
import com.st.stellar.pinmap.gpio.gpioDsl.Configuration
import com.st.stellar.pinmap.gpio.gpioDsl.Direction
import com.st.stellar.pinmap.gpio.gpioDsl.Family
import com.st.stellar.pinmap.gpio.gpioDsl.LQFPPackage
import com.st.stellar.pinmap.gpio.gpioDsl.NotConnectedPin
import com.st.stellar.pinmap.gpio.gpioDsl.Package
import com.st.stellar.pinmap.gpio.gpioDsl.Pin
import com.st.stellar.pinmap.gpio.gpioDsl.PinConfig
import com.st.stellar.pinmap.gpio.gpioDsl.SpecialPin
import com.st.stellar.pinmap.gpio.gpioDsl.SystemFunctionPin
import com.st.stellar.pinmap.gpioCfg.Board
import com.st.stellar.pinmap.gpioCfg.PinConfiguration
import com.st.stellar.pinmap.gpioCfg.PinSetting
import com.st.stellar.pinmap.gpioCfg.PinSettings
import com.st.stellar.pinmap.gpiocfg.glsp.server.GpioCfgModelTypes
import com.st.stellar.pinmap.helpers.GpioCfgHelper
import com.st.stellar.pinmap.validation.GpioValidationMessageAcceptor
import com.st.stellar.pinmap.validation.GpiocfgValidator
import java.util.ArrayList
import java.util.Collection
import java.util.List
import java.util.regex.Pattern
import org.eclipse.emf.common.util.Diagnostic
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EStructuralFeature
import org.eclipse.emf.ecore.util.Diagnostician
import org.eclipse.glsp.graph.GModelElement
import org.eclipse.glsp.graph.GModelRoot
import org.eclipse.glsp.graph.builder.impl.GLabelBuilder
import org.eclipse.glsp.graph.builder.impl.GNodeBuilder

class GeneratePinSettings {


	enum PinStatus {
		Configured,
		NotUsed,
		NotConnected,
		NotConfigurable
	}

	public val pinWidth = 24
	public val pinHeight = 12
	public val pinMarginWidth = 7
	public val pinMarginHeight = 3
	public val fontSize = pinWidth / 2
	public val generalOffsetX = -25
	public val generalOffsetY = -15

	@Inject extension GpioCfgHelper;

	static class PinSettingsErrorManager extends GpioValidationMessageAcceptor {
		public val Multimap<EObject, String> errors
		public val Multimap<EObject, String> infos
		public val Multimap<EObject, String> warnings

		new() {
			super(null)
			errors = HashMultimap.create
			infos = HashMultimap.create
			warnings = HashMultimap.create
			acceptsExternalResource = true
		}

		override collectError(String message, EObject object, EStructuralFeature feature, int index, String code,
			String... issueData) {
			errors.put(object, message)
		}

		override collectError(String message, EObject object, int offset, int length, String code,
			String... issueData) {
			errors.put(object, message)
		}

		override collectInfo(String message, EObject object, EStructuralFeature feature, int index, String code,
			String... issueData) {
			infos.put(object, message)
		}

		override collectInfo(String message, EObject object, int offset, int length, String code, String... issueData) {
			infos.put(object, message)
		}

		override collectWarning(String message, EObject object, EStructuralFeature feature, int index, String code,
			String... issueData) {
			warnings.put(object, message)
		}

		override collectWarning(String message, EObject object, int offset, int length, String code,
			String... issueData) {
			warnings.put(object, message)
		}

	}

	def dispatch String getRefId(PinSettings settings) {
		var n = (GpioCfgHelper.getBoard(settings.includedBoard).package.eContainer as Family).name
		n = n.replaceAll('x', GpioCfgHelper.getBoard(settings.includedBoard).package.name)
		var a = n.indexOf('_')
		n = n.substring(0, a)
		n
	}

	def dispatch String getRefId(PinSetting setting) {
		setting.pin.pinConfigurations.get(0).getRefId
	}

	def dispatch String getRefId(ConfigurablePin pin) {
		var cfg = pin.ref as Configuration
		cfg.name

	}

	new() {
		errors = new StringBuilder("errors:\n")
		warnings = new StringBuilder("warnings:\n")
	}

	var StringBuilder errors

	def error(String message) {
		errors.append("error: " + message + "\n")
	}

	var StringBuilder warnings

	def warning(String message) {
		warnings.append("warning: " + message + "\n")
	}

	def generateReport() {
		return errors.append(warnings)
	}

	def int getPortNb(String s) {
		val subString = s.charAt(1)
		var int port = Character.getNumericValue(subString)
		val begin = Character.getNumericValue('A')
		val int portNb = port - begin
		// println(portNb)
		portNb
	}

	def int getBit(String s) {
		val subString = s.substring(2, s.length)
		Integer.parseInt(subString)
	}

	def void generatePins(GModelRoot newRoot, GpioCfgModelState modelState, PinConfiguration pinConfiguration) {
		_gpioCfgHelper = GpioCfgHelper.instance

		val modules = GpioCfgHelper.getAllModules(pinConfiguration.pinSetting)
		var List<PinSetting> configuredPins = newArrayList
		if (modules.size > 0) {
			configuredPins = modules.flatMap[pins].toList
		}

		val pack = pinConfiguration.pinSetting.pack
		generatePackageLQFP(newRoot, modelState, pack, configuredPins, "")
	}

	def GModelElement generateLegend(GModelElement parent, GpioCfgModelState modelState, int width) {
		_gpioCfgHelper = GpioCfgHelper.instance
		val node = new PinNodeBuilder(GpioCfgModelTypes.LEGEND).position(width + pinWidth / 2, -50).id("legend")
		node.addArgument("width", width)
		val state = getCurrentLegendState(modelState)
		node.addArgument("state", state)

		val res = node.build()
		parent.children.add(res)

		res
	}
	
	def getCurrentLegendState(GpioCfgModelState modelState) {
		val key = modelState.getPinConfiguration().eResource().getURI().path()
        PreferenceManager.getPreference(key, "false")
	}
	

	def isBoardPin(PinSetting pin) {
		_gpioCfgHelper = GpioCfgHelper.instance
		if (pin !== null && pin.eContainer !== null && pin.eContainer.eContainer instanceof Board) {
			return true
		} else {
			return false
		}
	}

	static val directionNames = newLinkedHashMap(Direction.INPUT -> "in", Direction.OUTPUT -> "out",
		Direction.INPUT_OUTPUT -> "inout", Direction.INPUT_OR_OUTPUT -> "inout", Direction.GROUND -> "ground",
		Direction.POWER -> "power", Direction.INPUT_ALTERNATE -> "", Direction.OUTPUT_ALTERNATE -> "")

	def generatePackageLQFP(GModelRoot newRoot, GpioCfgModelState modelState, Package p,
		List<PinSetting> configuredPins, String additional) {

		var id = modelState.getIndex().indexEObject(p);

		val nbColumns = p.pins.size / 4
		val nbRows = nbColumns
		val pinSize = Math.min(pinWidth, pinHeight);
		val width = (pinSize + 2 * pinMarginHeight) * nbRows + pinMarginHeight
		val height = (pinSize + 2 * pinMarginHeight) * nbRows + pinMarginHeight

		val cssClasses = newArrayList
		cssClasses.add("gpiocfg-package")

		// val legendNode = generateLegend(newRoot, modelState, width);
		val packageNodeBuilder = new GNodeBuilder(GpioCfgModelTypes.PACKAGE).size(width, width)// .add(legendNode)
		.add(
			new GLabelBuilder(GpioCfgModelTypes.PACKAGE_NAME).text((p.eContainer as Family).label + p.name).id(
				"packageId").addCssClasses(cssClasses).position(width / 2, height / 2).build()).id(id).size(width,
			width)

		val packageNode = packageNodeBuilder.build();
		generateLegend(packageNode, modelState, width);

		newRoot.children.add(packageNode);

		val errorsMap = generateErrors(newRoot, modelState, configuredPins)
		for (pin : p.pins) {
			val config = pin.pinConfigurations.get(0) as PinConfig
			if (!(config instanceof NotConnectedPin)) {
				generatePinLQFP(newRoot, modelState, configuredPins, pin, errorsMap)
			}
		}
	}

	def Multimap<URI, PinSetting> getResources(Collection<PinSetting> settings) {
		val resources = HashMultimap.create()
		if (settings.size > 1) {
			for (d : settings) {
				val pinResourceUri = d.eResource.URI
				resources.put(pinResourceUri, d)

				var board = GpioCfgHelper.getBoard(d)
				if (board !== null) {
					val boardUri = board.eResource.URI
					if (!boardUri.equals(pinResourceUri)) {
						resources.put(pinResourceUri, d)
					}
				}
			}
		} else if (settings.size > 0) {
			val d = settings.get(0)
			resources.put(d.eResource.URI, d)
		}
		return resources
	}

	def generatePinLQFP(GModelRoot newRoot, GpioCfgModelState modelState, List<PinSetting> configuredPins, Pin pin,
		PinSettingsErrorManager errorsMap) {
		val ps = configuredPins.findFirst[it|it.pin !== null && it.pin.name.equals(pin.name)]
		if (ps === null) {
			generateNotConfiguredPinLQFP(newRoot, modelState, pin, errorsMap)
		} else {
			generateConfiguredPinLQFP(newRoot, modelState, ps, errorsMap)
		}
	}

	def generateNotConfiguredPinLQFP(GModelRoot newRoot, GpioCfgModelState modelState, Pin pin,
		PinSettingsErrorManager errorsMap) {

		val cssClasses = newArrayList
		val nodeClasses = newArrayList
		cssClasses.add("pinNone")
		nodeClasses.add("pinNone")

		val config = pin.pinConfigurations.get(0) as PinConfig
		var x = 0
		var y = 0
		val pattern = Pattern.compile("^.*_([0-9]+)$")
		val id = modelState.getIndex().indexEObject(pin); // pin.name
		var matcher = pattern.matcher(pin.name)
		if (matcher.find()) {
			val p = pin.eContainer as LQFPPackage
			val size = p.pins.size / 4

			var number = matcher.group(1)
			val index = Integer.parseInt(number)

			var side = ""
			if (index <= size) { // west
				side = "west"
			} else if (index <= size * 2 && index > size) { // south
				side = "south"
			} else if (index <= size * 3 && index > size * 2) { // east
				side = "east"
			} else if (index > size * 3) { // north
				side = "north"
			}

			var PinNodeBuilder pinNode = null

			var GLabelBuilder gName = null
			var GLabelBuilder gLabel = null
			var GLabelBuilder gDirection = null
			pinNode = new PinNodeBuilder().id(id)
			pinNode.addArgument("side", side)
			pinNode.addArgument("name", pin.name)
			gLabel = null
			if (SpecialPin.isInstance(config)) {
				val sp = SpecialPin.cast(config)
				val directionName = GeneratePinSettings.directionNames.get(sp.direction) + "_" + side

				nodeClasses.add("nonConfigurablePin")
				gName = new GLabelBuilder(GpioCfgModelTypes.PIN_NAME).text(sp.ref.name).id(id + "_pin_name")
				gName.addArgument("side", side)
				gLabel = new GLabelBuilder(GpioCfgModelTypes.PIN_ID).text(pin.pinName).id(id + "_pin_label")
				gLabel.addArgument("side", side)
				gDirection = new GLabelBuilder(GpioCfgModelTypes.PIN_DIRECTION).id(id + "_pin_direction")
				gDirection.addArgument("direction", directionName)
			} else if (SystemFunctionPin.isInstance(config)) {
				val sp = SystemFunctionPin.cast(config)

				val directionName = GeneratePinSettings.directionNames.get(sp.direction) + "_" + side
				nodeClasses.add("nonConfigurablePin")

				gName = new GLabelBuilder(GpioCfgModelTypes.PIN_NAME).text(sp.ref.name).id(id + "_pin_name")
				gName.addArgument("side", side)

				gLabel = new GLabelBuilder(GpioCfgModelTypes.PIN_ID).text(pin.pinName).id(id + "_pin_label")
				gLabel.addArgument("side", side)

				gDirection = new GLabelBuilder(GpioCfgModelTypes.PIN_DIRECTION).id(id + "_pin_direction")
				gDirection.addArgument("direction", directionName)
			} else if (NotConnectedPin.isInstance(config)) {
				gName = new GLabelBuilder(GpioCfgModelTypes.PIN_NAME).text("not connected").id(id + "_pin_name")
				gName.addArgument("side", side)

			} else if (ConfigurablePin.isInstance(config)) {
				gLabel = new GLabelBuilder(GpioCfgModelTypes.PIN_ID).text(pin.pinName).id(id + "_pin_label")
				gLabel.addArgument("side", side)
			}

			var imageOffsetX = 0
			var imageOffsetY = 0
			// println(index)
			if (index <= size) { // west
				x = 0
				y = index
				x *= (pinWidth + 2 * pinMarginWidth)
				x += -3
				y *= (pinHeight + 2 * pinMarginHeight)
				imageOffsetY = 1
				imageOffsetX = - pinWidth / 2
				side = "west"

				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index <= size * 2 && index > size) { // south
				y = (size + 1) * (pinHeight + 2 * pinMarginHeight) + pinMarginHeight
				y += 1
				x = index - size
				x *= (pinHeight + 2 * pinMarginHeight)
				x += pinHeight
				// y *= (pinHeight + 2 * pinMarginWidth)
				// y += pinHeight - pinMarginHeight
				imageOffsetX = 0
				imageOffsetY += pinWidth + 2
				side = "south"
				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).position(pinHeight + pinMarginHeight, 0).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index <= size * 3 && index > size * 2) { // east
				x = size + 2
				y = size * 3 - index + 1
				x *= (pinHeight + 2 * pinMarginHeight)
				x -= 4
				y *= (pinHeight + 2 * pinMarginHeight)
				imageOffsetX = pinWidth + 2
				imageOffsetY = 1
				side = "east"

				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).position(pinWidth + pinMarginWidth, 0).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index > size * 3) { // north
				x = (size * 4) - index + 1
				y = 1

				x *= (pinHeight + 2 * pinMarginHeight)
				x += pinHeight

				y *= (pinHeight + 2 * pinMarginHeight)
				y -= 2 * (pinHeight + pinMarginHeight)
				y -= 1
				imageOffsetX = 0
				imageOffsetY = -pinHeight - 2 // (pinWidth + pinHeight + pinMarginWidth)
				side = "north"
				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			}

			nodeClasses.add(side)
			x += generalOffsetX
			y += generalOffsetY

			newRoot.getChildren().add(pinNode.addCssClasses(nodeClasses).position(x, y).build)
		}
	}

	def generateConfiguredPinLQFP(GModelRoot newRoot, GpioCfgModelState modelState, PinSetting ps,
		PinSettingsErrorManager errorsMap) {

		val pin = ps.pin
		val cssClasses = newArrayList
		val nodeClasses = newArrayList
		if (ps.isBoardPin) {
			cssClasses.add("board")
			nodeClasses.add("board")
		} else {
			cssClasses.add("cPin")
			nodeClasses.add("cPin")
		}

		var x = 0
		var y = 0
		val pattern = Pattern.compile("^.*_([0-9]+)$")
		val id = modelState.getIndex().indexEObject(pin); // pin.name
		var matcher = pattern.matcher(pin.name)
		if (matcher.find()) {
			val p = pin.eContainer as LQFPPackage
			val size = p.pins.size / 4

			var number = matcher.group(1)
			val index = Integer.parseInt(number)

			var side = ""
			if (index <= size) { // west
				side = "west"
			} else if (index <= size * 2 && index > size) { // south
				side = "south"
			} else if (index <= size * 3 && index > size * 2) { // east
				side = "east"
			} else if (index > size * 3) { // north
				side = "north"
			}

			var PinNodeBuilder pinNode = null

			var GLabelBuilder gName = null
			var GLabelBuilder gLabel = null
			var GLabelBuilder gDirection = null
			val directionName = ps.config.direction.literal + "_" + side

			gName = new GLabelBuilder(GpioCfgModelTypes.PIN_NAME).text(ps.name).id(id + "_pin_name")
			gName.addArgument("side", side)

			gLabel = new GLabelBuilder(GpioCfgModelTypes.PIN_ID).text(pin.pinName).id(id + "_pin_label")
			gLabel.addArgument("side", side)

			gDirection = new GLabelBuilder(GpioCfgModelTypes.PIN_DIRECTION).id(id + "_pin_direction")
			gDirection.addArgument("direction", directionName)

			val errors = errorsMap.errors.get(ps)
			val warnings = errorsMap.warnings.get(ps)
			val infos = errorsMap.infos.get(ps)
			if (errors.size > 0 || warnings.size > 0 || infos.size > 0) {
				pinNode = new PinNodeBuilder(GpioCfgModelTypes.PIN_ERROR).id(id)
				pinNode.addArgument("name", pin.name)
				if (errors.size > 0) {
					pinNode.addArgument("error", errors.join(","))
					nodeClasses.add("error");
					cssClasses.add("error");
				} else if (warnings.size > 0) {
					pinNode.addArgument("warning", warnings.join(","))
					nodeClasses.add("warning");
					cssClasses.add("warning");
				} else if (infos.size > 0) {
					pinNode.addArgument("info", infos.join(","))
					nodeClasses.add("info");
					cssClasses.add("info");
				}
				pinNode.addArgument("side", side)
			} else {
				pinNode = new PinNodeBuilder().id(id)
				pinNode.addArgument("side", side)
				pinNode.addArgument("name", pin.name)
			}

			var imageOffsetX = 0
			var imageOffsetY = 0
			// println(index)
			if (index <= size) { // west
				x = 0
				y = index
				x *= (pinWidth + 2 * pinMarginWidth)
				x += -3
				y *= (pinHeight + 2 * pinMarginHeight)
				imageOffsetY = 1
				imageOffsetX = - pinWidth / 2
				side = "west"

				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index <= size * 2 && index > size) { // south
				y = (size + 1) * (pinHeight + 2 * pinMarginHeight) + pinMarginHeight
				y += 1
				x = index - size
				x *= (pinHeight + 2 * pinMarginHeight)
				x += pinHeight
				// y *= (pinHeight + 2 * pinMarginWidth)
				// y += pinHeight - pinMarginHeight
				imageOffsetX = 0
				imageOffsetY += pinWidth + 2
				side = "south"
				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).position(pinHeight + pinMarginHeight, 0).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index <= size * 3 && index > size * 2) { // east
				x = size + 2
				y = size * 3 - index + 1
				x *= (pinHeight + 2 * pinMarginHeight)
				x -= 4
				y *= (pinHeight + 2 * pinMarginHeight)
				imageOffsetX = pinWidth + 2
				imageOffsetY = 1
				side = "east"

				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).position(pinWidth + pinMarginWidth, 0).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index > size * 3) { // north
				x = (size * 4) - index + 1
				y = 1

				x *= (pinHeight + 2 * pinMarginHeight)
				x += pinHeight

				y *= (pinHeight + 2 * pinMarginHeight)
				y -= 2 * (pinHeight + pinMarginHeight)
				y -= 1
				imageOffsetX = 0
				imageOffsetY = -pinHeight - 2 // (pinWidth + pinHeight + pinMarginWidth)
				side = "north"
				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			}

			nodeClasses.add(side)
			x += generalOffsetX
			y += generalOffsetY

			newRoot.getChildren().add(pinNode.addCssClasses(nodeClasses).position(x, y).build)
		}
	}

	def EObject getEObject(
		GpioCfgModelState modelState,
		GModelElement element
	) {
		val selectedPinConfigs = modelState.index.getEObject(element.id)

		val ps = selectedPinConfigs.stream().filter(it|EObject.isInstance(it)).map(it|EObject.cast(it)).findFirst();
		return ps.orElse(null);
	}

	def getPinName(Pin pin) {
		val pattern = Pattern.compile("^(.*)_([0-9]+)$")
		val id = pin.name
		var matcher = pattern.matcher(id)
		if (matcher.find()) {
			var number = matcher.group(2)
			return number
		} else {
			return ""
		}
	}

	def boolean relatedToPinSetting(Diagnostic diag, PinSetting ps) {
		if(ps === null) return false
		if(ps.config === null) return false
		if(ps.ioFields === null) return false

		var res = false
		val List<EObject> objList = newArrayList
		objList.add(ps)
		objList.add(ps.config)
		objList.add(ps.ioFields)
		for (reg : ps.ioFields.reg) {
			objList.add(reg)
		}
		if (objList.contains(diag.data.get(0))) {
			res = true
		}
		return res
	}

	def PinSettingsErrorManager generateErrors(GModelRoot newRoot, GpioCfgModelState modelState,
		List<PinSetting> configuredPins) {

		val pinConfiguration = modelState.pinConfiguration
		val errorManager = new PinSettingsErrorManager()
		GpiocfgValidator.checkNoDuplicatedPinNames(pinConfiguration, errorManager)
		GpiocfgValidator.checkUniqueFunctionUsage(pinConfiguration, errorManager)
		GpiocfgValidator.checkPinAlreadyUsed(pinConfiguration, errorManager)

		for (ps : configuredPins) {
			val diags = new ArrayList<Diagnostic>();

			GpiocfgValidator.checkMissingRegInstances(ps, errorManager)

			if (ps.isBoardPin) {
				val errorCfg = Diagnostician.INSTANCE.validate(pinConfiguration);
				val errorsCfg = errorCfg.getChildren();
				diags.addAll(errorsCfg.filter[it|it.relatedToPinSetting(ps)]);
				for (Diagnostic e : errorsCfg) {
					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
				}
			} else {
				val errorPs = Diagnostician.INSTANCE.validate(ps);
				val errorsPs = errorPs.getChildren();
				diags.addAll(errorsPs.filter[it|it.relatedToPinSetting(ps)]);
				for (Diagnostic e : errorsPs) {
					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
				}
			}
			if (diags.size > 0) {
				diags.filter[d|d.severity == Diagnostic.ERROR].map(it|it.message).forEach(
					it |
						errorManager.errors.put(ps, it)
				)
				diags.filter[d|d.severity == Diagnostic.WARNING].map(it|it.message).forEach(
					it |
						errorManager.warnings.put(ps, it)
				)
				diags.filter[d|d.severity == Diagnostic.INFO].map(it|it.message).forEach(
					it |
						errorManager.infos.put(ps, it)
				)
			}
		}
		return errorManager
	}

}
