package com.st.stellar.pinmap.gpiocfg.htmlGenerator.runtime

import com.google.common.collect.HashMultimap
import com.google.common.collect.Multimap
import com.google.inject.Inject
import com.google.inject.Provider
import com.st.stellar.pinmap.gpio.gpioDsl.Family
import com.st.stellar.pinmap.gpioCfg.Board
import com.st.stellar.pinmap.gpioCfg.PinConfiguration
import com.st.stellar.pinmap.gpioCfg.PinSetting
import com.st.stellar.pinmap.gpioCfg.PinSettings
import com.st.stellar.pinmap.helpers.GpioCfgHelper
import java.util.ArrayList
import java.util.List
import java.util.Locale
import org.eclipse.core.resources.IProject
import org.eclipse.core.resources.IResource
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.CoreException
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.emf.common.util.Diagnostic
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.resource.ResourceSet
import org.eclipse.emf.ecore.util.Diagnostician
import org.eclipse.xtext.generator.AbstractGenerator
import org.eclipse.xtext.generator.IFileSystemAccess2
import org.eclipse.xtext.generator.IGeneratorContext
import com.st.stellar.pinmap.validation.GpiocfgValidator
import java.util.Map

class GpioCfgHtmlGenerator extends AbstractGenerator {

	@Inject Provider<ResourceSet> rsp
	@Inject extension GpioCfgHelper

	override void doGenerate(Resource resource, IFileSystemAccess2 fsa, IGeneratorContext context) {

		val resourceSet = rsp.get
		val r = resourceSet.getResource(resource.URI, true)
		if(r.contents.empty) return
		val configuration = r.contents.head as PinConfiguration

		// Ugly code to check whether we need to generate HTML code or not
		// TODO: find a more elegant code
		val boolean generateHtml = context.cancelIndicator.class.name.contains("GenerateHTMLDiagramCancelIndicator")

		if (generateHtml) {

			if (configuration.pinSetting instanceof PinSettings) {
				val pinSettings = configuration.pinSetting as PinSettings
				val imageHtml = generateHTML(configuration)

				val filename = pinSettings.getGenerationFileName("html")
				fsa.generateFile(filename, imageHtml)

				updateExplorer(configuration);
			}
		}

	}

	def String generateHTML(PinConfiguration cfg) {

		if (!(cfg.pinSetting instanceof PinSettings))
			return null

		val pinSettings = cfg.pinSetting as PinSettings

		if (GpioCfgHelper.getBoard(pinSettings.includedBoard).package === null)
			return null

		var n = (GpioCfgHelper.getBoard(pinSettings.includedBoard).package.eContainer as Family).name
		n = n.replaceAll('x', pinSettings.name)

		val gen = new GeneratePinSettings

		val modules = GpioCfgHelper.getAllModules(cfg.pinSetting)
		var List<PinSetting> configuredPins = newArrayList
		if (modules.size > 0) {
			configuredPins = modules.flatMap[it|it.pins].toList
		}
		val errorsMap = generateErrors(cfg, configuredPins)

		val imageHtml = gen.generateView(pinSettings, errorsMap)
		return imageHtml
	}

	def void checkNoDuplicatedPinNames(PinConfiguration cfg, Multimap<String, String> errors) {
		val multiMap = HashMultimap.create()

		if (cfg.board !== null) {
			for (e : GpioCfgHelper.getPins(cfg.board)) {
				multiMap.put(e.name.toLowerCase(Locale.ENGLISH), e)
			}
		}
		if (cfg.pinSetting !== null) {
			for (e : GpioCfgHelper.getAllPinSettings(cfg)) {
				multiMap.put(e.name.toLowerCase(Locale.ENGLISH), e)
			}
		}

		for (entry : multiMap.asMap.entrySet) {
			val duplicates = entry.value
			if (duplicates.size > 1) {
				for (d : duplicates) {
					errors.put(d.pin.name, "Duplicated pin identifier: " + d.name)
				}
			}
		}
	}

	def Map<EObject, Map<Integer, List<String>>> generateErrors(PinConfiguration pinConfiguration, List<PinSetting> configuredPins) {
		val errorManager = new PinSettingsErrorManager()
		GpiocfgValidator.checkNoDuplicatedPinNames(pinConfiguration, errorManager)
		GpiocfgValidator.checkUniqueFunctionUsage(pinConfiguration, errorManager)
		GpiocfgValidator.checkPinAlreadyUsed(pinConfiguration, errorManager)

		val errorsMap = newHashMap
		for (ps : configuredPins) {

			val diags = new ArrayList<Diagnostic>();

			GpiocfgValidator.checkMissingRegInstances(ps, errorManager)

			val pin = ps.pin
			var Map<Integer, List<String>> psMap = errorsMap.get(pin) as Map<Integer, List<String>>
			if (psMap === null) {
				psMap = newHashMap
				errorsMap.put(pin, psMap)
			}
			var infosList = psMap.get(Integer.valueOf(Diagnostic.INFO))
			if (infosList === null) {
				infosList = newArrayList
				psMap.put(Integer.valueOf(Diagnostic.INFO), infosList)
			}
			var warningsList = psMap.get(Integer.valueOf(Diagnostic.WARNING))
			if (warningsList === null) {
				warningsList = newArrayList
				psMap.put(Integer.valueOf(Diagnostic.WARNING), warningsList)
			}
			var errorsList = psMap.get(Integer.valueOf(Diagnostic.ERROR))
			if (errorsList === null) {
				errorsList = newArrayList
				psMap.put(Integer.valueOf(Diagnostic.ERROR), errorsList)
			}

			if (ps.isBoardPin) {
				val errorCfg = Diagnostician.INSTANCE.validate(pinConfiguration);
				val errorsCfg = errorCfg.getChildren();
				diags.addAll(errorsCfg.filter[it|it.relatedToPinSetting(ps)]);
				for (Diagnostic e : errorsCfg) {
					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
				}
				val errorPs = Diagnostician.INSTANCE.validate(ps);
				val errorsPs = errorPs.getChildren();
				diags.addAll(errorsPs.filter[it|it.relatedToPinSetting(ps)]);
				for (Diagnostic e : errorsPs) {
					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
				}
			} else {
				val errorPs = Diagnostician.INSTANCE.validate(ps);
				val errorsPs = errorPs.getChildren();
				diags.addAll(errorsPs.filter[it|it.relatedToPinSetting(ps)]);
				for (Diagnostic e : errorsPs) {
					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
				}
			}

			if (diags.size > 0) {
				diags.filter[d|d.severity == Diagnostic.ERROR].map(it|it.message).forEach(
					msg |
						errorManager.errors.put(ps, msg)
				)
				diags.filter[d|d.severity == Diagnostic.WARNING].map(it|it.message).forEach(
					msg |
						errorManager.warnings.put(ps, msg)
				)
				diags.filter[d|d.severity == Diagnostic.INFO].map(it|it.message).forEach(
					msg |
						errorManager.infos.put(ps, msg)
				)
			}
			val errors = errorManager.errors.get(ps)
			if (errors.size > 0) {
				errors.forEach[msg|errorsMap.get(ps.pin).get(Integer.valueOf(Diagnostic.ERROR)).add(msg)]
			}
			val warnings = errorManager.warnings.get(ps)
			if (warnings.size > 0) {
				warnings.forEach[msg|errorsMap.get(ps.pin).get(Integer.valueOf(Diagnostic.WARNING)).add(msg)]
			}
		}

		return errorsMap
//		val Multimap<String, String> errors = HashMultimap.create
//
//		for (ps : configuredPins) {
//			val diags = new ArrayList<Diagnostic>();
//
//			checkNoDuplicatedPinNames(pinConfiguration, errors)
//			if (ps.isBoardPin) {
//				val errorCfg = Diagnostician.INSTANCE.validate(pinConfiguration);
//				val errorsCfg = errorCfg.getChildren();
//				diags.addAll(errorsCfg.filter[it|it.relatedToPinSetting(ps)]);
//				for (Diagnostic e : errorsCfg) {
//					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
//				}
//			} else {
//				val errorPs = Diagnostician.INSTANCE.validate(ps);
//				val errorsPs = errorPs.getChildren();
//				diags.addAll(errorsPs.filter[it|it.relatedToPinSetting(ps)]);
//				for (Diagnostic e : errorsPs) {
//					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
//				}
//			}
//			if (diags.size > 0) {
//				diags.map(it|it.message).forEach(it|errors.put(ps.pin.name, it))
//			}
//		}
//		return errors
	}

	def isBoardPin(PinSetting pin) {
		_gpioCfgHelper = GpioCfgHelper.instance
		if (pin.eContainer.eContainer instanceof Board) {
			return true
		} else {
			return false
		}
	}

	def boolean relatedToPinSetting(Diagnostic diag, PinSetting ps) {
		var res = false
		val List<EObject> objList = newArrayList
		objList.add(ps)
		objList.add(ps.config)
		objList.add(ps.ioFields)
		for (reg : ps.ioFields.reg) {
			objList.add(reg)
		}
		if (objList.contains(diag.data.get(0))) {
			res = true
		}
		return res
	}

	static def IProject getProject(EObject model) {
		var IProject project = null
		val uri = model.eResource().getURI()
		if (!uri.isFile()) {
			project = null;
		}
		val projectName = uri.segmentsList().get(1).toString();
		project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		return project;
	}

	static def void updateExplorer(EObject model) {

		val project = getProject(model);
		if (project !== null) {
			try {
				project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor())
			} catch (CoreException e) {
				// TODO Auto-generated catch block
				e.printStackTrace()
			}
		}
	}
}
