package com.st.stellar.pinmap.pincfg.codeGenerator.runtime

import com.google.inject.Inject
import com.google.inject.Provider
import com.st.stellar.pinmap.helpers.PinCfgHelper
import com.st.stellar.pinmap.pinCfg.LatchedState
import com.st.stellar.pinmap.pinCfg.PinConfiguration
import com.st.stellar.pinmap.pinCfg.PinSetting
import com.st.stellar.pinmap.pinCfg.PinSettings
import com.st.stellar.pinmap.pinmapDsl.PinmapModel
import java.util.ArrayList
import java.util.Locale
import java.util.regex.Pattern
import org.eclipse.core.runtime.Platform
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.resource.ResourceSet
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.generator.AbstractGenerator
import org.eclipse.xtext.generator.IFileSystemAccess
import org.eclipse.xtext.generator.IFileSystemAccess2
import org.eclipse.xtext.generator.IGeneratorContext
import org.eclipse.xtext.generator.JavaIoFileSystemAccess

// Implements SDD_PINCFG_CODE_GENERATION
class PinCfgCodeGenerator extends AbstractGenerator {

	@Inject Provider<ResourceSet> rsp
	@Inject extension PinCfgHelper

	static int pageWidth = 80;

	override void doGenerate(Resource resource, IFileSystemAccess2 fsa, IGeneratorContext context) {
		val resourceSet = rsp.get
		val r = resourceSet.getResource(resource.URI, true)
		if(r.contents.empty) return

		val head = r.contents.head
		val cfg = head as PinConfiguration
		if (cfg.pinSetting instanceof PinSettings) {
			val pinSettings = cfg.pinSetting

			var outputFolder = ""
			if (pinSettings.outputFolder !== null) {
				outputFolder = "../" + pinSettings.outputFolder + "/"
			} else {
				if (fsa instanceof JavaIoFileSystemAccess) {
					// We are generating from headless generator
					val fsa2 = fsa as JavaIoFileSystemAccess
					val outputCfg = fsa2.outputConfigurations.get("PinCfgCode")
						
					outputFolder = outputCfg.getOutputDirectory() + "/"
					outputFolder = '../' + outputFolder
				} else {
					// We are generating from Eclipse generator
					outputFolder = "./src-gen/"
					outputFolder = '../' + outputFolder
				}
			}

			val bootheader = generatePinSettings(pinSettings)
			// fsa.generateFile(outputFolder + pinSettings.name + ".c", bootheader)
			fsa.generateFile(outputFolder + pinSettings.name + ".h", IFileSystemAccess.DEFAULT_OUTPUT, bootheader)
			/*try {
				ResourcesPlugin.workspace.root.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor)
			} catch (IllegalStateException ise) {
				// Ignore it... workspace is closed
			}*/
		}
	}

	static String PINMAPCFG_BUNDLE_ID = "com.st.stellar.pinmap.pincfg"

	static def String getVersion() {
		if (Platform.running) {
			val ideBundle = Platform.getBundle(PINMAPCFG_BUNDLE_ID)
			ideBundle.getVersion().toString
		} else {
			PinCfgHelper.PINCFG_VERSION
		}
	}

	static def String getPluginName() {
		if (Platform.running) {
			val ideBundle = Platform.getBundle(PINMAPCFG_BUNDLE_ID)
			ideBundle.getSymbolicName()
		} else {
			PINMAPCFG_BUNDLE_ID
		}
	}
	
	def generatePinSettings(PinSettings pinSettings) {

		EcoreUtil.resolveAll(pinSettings.eResource)
		var writtenPins = new ArrayList<PinSetting>
		var boardName = ((pinSettings.eContainer as PinConfiguration).family.eContainer as PinmapModel).name
		boardName = boardName.substring(boardName.lastIndexOf('.')+1)

		var fieldStr1 = '''
			/****************************************************************************
			*
			* Copyright (c) 2024-2026 STMicroelectronics - All Rights Reserved
			*
			* License terms: STMicroelectronics Proprietary in accordance with licensing
			* terms SLA0098 at www.st.com.
			*
			* THIS SOFTWARE IS DISTRIBUTED "AS IS," AND ALL WARRANTIES ARE DISCLAIMED,
			* INCLUDING MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
			*
			* PLEASE NOTE THAT THIS FILE IS GENERATED FROM A CONFIGURATION FILE,
			* USING «pluginName», Version «version.toString»
			* MODIFYING IT IMPLIES POTENTIAL LOSS OF YOUR MODIFICATIONS.
			*
			*****************************************************************************/
			
			/**
			 * @file    «pinSettings.name.toLowerCase(Locale.US)».h
			 * @brief   «pinSettings.name.toLowerCase(Locale.US)» board header.
			 *
			 * @defgroup PLATFORM PLATFORM
			 * @defgroup BOARD BOARD
			 * @ingroup PLATFORM
			 * @addtogroup «pinSettings.familyLabel»
			 * @ingroup BOARD
			 * @{
			 */
			
			#ifndef _«pinSettings.name.toUpperCase(Locale.US)»_H_
			#define _«pinSettings.name.toUpperCase(Locale.US)»_H_
			
			#define «PinCfgHelper.getBoard(pinSettings).name.toUpperCase(Locale.US)» 1

			#include <siul2.h>
			
			/*===========================================================================*/
			/* Board defines.                                                            */
			«pinSettings.familyDescriptionComment»
			/*===========================================================================*/
			
			#define «pinSettings.name.toUpperCase(Locale.US)» 1
			
			«FOR mod : PinCfgHelper.getAllModules(pinSettings)»
				«IF mod.comment === null»
					/*--------------------- «mod.name.name» pins definitions ---------------------*/
				«ELSE»
					/*
					 * «mod.comment»
					 */
				«ENDIF»
				«FOR pin : mod.pins»
					«IF writtenPins.filter[name.equals(pin.name)].size == 0»
						«val addRes = writtenPins.add(pin)»
						«IF pin.config.alt.label.startsWith("EIRQ")»
						#define «pin.toUpper»_NUMBER          SIUL2_«pin.config.alt.label»
						«ENDIF»
						/* «pin.getPadName» - MSCR_IO«pin.mscrIndex» */
						#define «pin.toUpper» siul2_iopack(«pin.portName», «pin.pinPad»)
						«pin.getIODefinitions»
						«IF pin.isSetLatchedState»
						#define «pin.toUpper»_LATCHED_STATE		«pin.latchedState==LatchedState.HIGH?"SIUL2_IO_HIGH":"SIUL2_IO_LOW"»
						«ENDIF»
						«pin.getMuxDefinition»


					«ENDIF»
				«ENDFOR»
			«ENDFOR»
		'''
		writtenPins = new ArrayList<PinSetting>
		var fieldStr2 = '''
			/*===========================================================================*/
			/* External declarations.                                                    */
			/*===========================================================================*/
			
			#ifdef __cplusplus
			extern "C" {
			#endif
			
			#define «pinSettings.name»_init() \
			    do {                                                                      \
				«FOR mod : PinCfgHelper.getAllModules(pinSettings)»
					«val comment = '''/* «mod.name.name» */'''»
					    «comment»«" ".repeat(pageWidth - Math.max(comment.length,1)-10)»\
					«FOR pin : mod.pins»
						«IF writtenPins.filter[name.equals(pin.name)].size == 0»
						«val addRes = writtenPins.add(pin)»
						«val ioMode = '''    siul2_set_iomode(«pin.toUpper», «pin.toUpper»_CFG);'''»
						«var ioSpaces = pageWidth - ioMode.length -6»
						«ioMode»«" ".repeat(Math.max(ioSpaces,1))»\
						«val muxMode =  '''    siul2_set_muxmode(«pin.toUpper»_IOMUX, «pin.toUpper»_IOMUX_CFG);'''»
						«val muxSpaces = pageWidth - muxMode.length-6»
						«IF pin.hasMUX && (pin.functionRefMux !== null)»
							«muxMode»«" ".repeat(Math.max(muxSpaces,1))»\
						«ENDIF»
						«val latchedState = pin.isSetLatchedState?'''    siul2_iowrite(«pin.toUpper», «pin.toUpper»_LATCHED_STATE);''':""»
						«val latchedSpaces = pageWidth - latchedState.length -6»
						«latchedState»«" ".repeat(Math.max(latchedSpaces,1))»\
						«" ".repeat(pageWidth-6)»\
						«ENDIF»«ENDFOR»
					«ENDFOR»
				} while (0)
				
			#ifdef __cplusplus
			}
			#endif
			
			#endif /* _«pinSettings.name.toUpperCase(Locale.US)»_H_ */
			
			/** @} */
					
		'''
		fieldStr1 + fieldStr2
	}

	def String toUpper(PinSetting ps) {
		ps.name.toUpperCase(Locale.US)
	}

	def String getFamilyDescriptionComment(PinSettings ps) {
		var StringBuffer sb = new StringBuffer
		val desc = ps.familyDescription
		if (desc.length > 0) {
			sb.append("/*                                                                           */\n")
			for (line : desc.split('\n')) {
				val l = line.trim.replaceAll("\n", "")
				sb.append("/* " + l + (l.length > 73 ? "" : " ".repeat(73 - l.length)) + " */\n")
			}
			sb.append("/*                                                                           */\n")
		}
		sb.toString
	}

	def String getIODefinitions(PinSetting ps) {
		var ioDef = ""
		val spaces = pageWidth - ps.toUpper.length - "_CFG".length - "#define ".length -2
		ioDef = '''
			
			#define «ps.toUpper»_CFG«" ".repeat(spaces)»\
				/* «ps.config.toIOHex» : «ps.config.IOValues»
		'''
		ioDef
	}

	def boolean isMUX(PinSetting ps) {
		val funcRef = ps.functionRef
		if (funcRef !== null && funcRef.alt !== null && funcRef.alt.ref !== null) {
			if (funcRef.alt.input === null && funcRef.alt.inputOutput === null && funcRef.alt.output === null) {
				// LOW: StellarERCR #1345515: P7 generate useless IOMUX values
				return funcRef.alt.sss > 511
				// )
			}
		}
		false
	}

	def String getMuxDefinition(PinSetting ps) {

		var muxDef = ""
		if (ps.isMUX) {
			val funcRef = ps.functionRef
			muxDef = '''
				
				#define «ps.toUpper»_IOMUX              «funcRef.alt.index»u
				#define «ps.toUpper»_IOMUX_CFG          «Long.parseLong(ps.config.toMUXHex.substring(2), 16)»u
			'''
		} else if (ps.hasMUX) {
			val funcRefMux = ps.functionRefMux
			if (funcRefMux !== null) {
				muxDef = '''
					
					#define «ps.toUpper»_IOMUX             «funcRefMux.alt.index»u
					#define «ps.toUpper»_IOMUX_CFG         «Long.parseLong(ps.config.toMUXHex(funcRefMux).substring(2), 16)»u
				'''
			}
		} else {
			muxDef = '''
			'''
		}
		muxDef
	}

	def getPadName(PinSetting setting) {
		val pad = setting.getPad()
		var pattern = Pattern.compile("P([^\\d]+)(\\d+)")
		var matcher = pattern.matcher(pad.name)
		if (matcher.find()) {
			return "P" + matcher.group(1) + "[" + matcher.group(2) + "]"
		}
		return pad.name
	}

	def getPinPad(PinSetting setting) { // SIUL2_PAD_0
		val pad = setting.getPad()
		var pattern = Pattern.compile("P([^\\d]+)(\\d+)")
		var matcher = pattern.matcher(pad.name)
		if (matcher.find()) {
			return "SIUL2_PAD_" + matcher.group(2)
		}
		return pad.name
	}

	def getPortName(PinSetting setting) { // SIUL2_PORT_D
		val pad = setting.getPad()
		var pattern = Pattern.compile("P([^\\d]+)(\\d+)")
		var matcher = pattern.matcher(pad.name)
		if (matcher.find()) {
			return "SIUL2_PORT_" + matcher.group(1)
		}
		return pad.name
	}

}
