package com.st.stellar.pinmap.pincfg.excelGenerator.runtime

import com.google.inject.Inject
import com.google.inject.Provider
import com.st.stellar.pinmap.helpers.PinCfgHelper
import com.st.stellar.pinmap.helpers.ResourceUtils
import com.st.stellar.pinmap.pinCfg.Include
import com.st.stellar.pinmap.pinCfg.PinConfiguration
import com.st.stellar.pinmap.pinCfg.PinSettings
import com.st.stellar.pinmap.pinmapDsl.ConfigurablePin
import com.st.stellar.pinmap.pinmapDsl.Family
import com.st.stellar.pinmap.pinmapDsl.NotConnectedPin
import com.st.stellar.pinmap.pinmapDsl.Pin
import com.st.stellar.pinmap.pinmapDsl.PinmapModel
import com.st.stellar.pinmap.pinmapDsl.SpecialPin
import com.st.stellar.pinmap.pinmapDsl.SystemFunctionPin
import java.io.Closeable
import java.io.File
import java.io.FileNotFoundException
import java.io.FileOutputStream
import java.io.IOException
import java.util.ArrayList
import java.util.List
import org.apache.poi.hssf.usermodel.HSSFWorkbook
import org.apache.poi.ss.usermodel.IndexedColors
import org.eclipse.core.resources.IProject
import org.eclipse.core.resources.IResource
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.CoreException
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.core.runtime.Path
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.resource.ResourceSet
import org.eclipse.xtext.generator.AbstractGenerator
import org.eclipse.xtext.generator.IFileSystemAccess2
import org.eclipse.xtext.generator.IGeneratorContext
import org.spiritconsorsium.xml.spiritmodel.Spirit.DocumentRoot
import org.spiritconsorsium.xml.spiritmodel.Spirit.MemoryMapType
import org.spiritconsorsium.xml.spiritmodel.Spirit.RegisterType

class ExcelGenerator extends AbstractGenerator {

	@Inject extension PinCfgHelper
	@Inject Provider<ResourceSet> rsp

	override void doGenerate(Resource r, IFileSystemAccess2 fsa, IGeneratorContext context) {

		val resourceSet = rsp.get
		val resource = resourceSet.getResource(r.URI, true)
		// Apply Excel generation
		if(resource.contents.head === null) return
		val configuration = resource.contents.head as PinConfiguration

		// Ugly code to check whether we need to generate HTML code or not
		// TODO: find a more elegant code
		val boolean generateExcel = context.cancelIndicator.class.name.contains("GenerateExcelFileCancelIndicator")

		if (generateExcel) {

			val pinSettings = configuration.pinSetting
			if(pinSettings === null) return;

			val incBoard = PinCfgHelper.getBoard(pinSettings.includedBoard) 
			if(incBoard.getPackage() === null) return
			var n = (incBoard.package.eContainer as Family).name
			n = n.replaceAll('x', pinSettings.name)

			// Implements SDD_GpioCfg_EXPORT_EXCEL
			// Excel generation
			// var fileName = "../data-gen/" + n + ".xls"
			val fileName = pinSettings.getGenerationFileName("xls")
			// val fileName = file.toString.replace('L/', '../data-gen/')
			// fsa.generateFile("../data-gen/" + n + ".txt", "")
			// fsa.deleteFile("../data-gen/" + n + ".txt")
			var excelFileUri = fsa.getURI(fileName, "PinCfgCode")
			val file2 = ResourcesPlugin.workspace.root.getFile(new Path(excelFileUri.toPlatformString(true)))
			generateExcel(file2.rawLocation.toOSString, pinSettings)

			updateExplorer(configuration);
		}
	}

	def static toText(Include imp) {
		val PinmapModel importedPinmap = ResourceUtils.resourceToEObject(ResourceUtils.openImport(imp.eResource,
			imp.importURI)) as PinmapModel;
		return '''
			Pinmap model file: «imp.importURI»:
			«IF importedPinmap === null»
				File cannot be opened.
			«ELSE»
				Number of families: «importedPinmap.families.size»
				«FOR imp2 : importedPinmap.includes»
					«toText(imp2)»
				«ENDFOR»
			«ENDIF»
		''';
	}

	static def List<RegisterType> registers(DocumentRoot root) {
		var registerTypes = new ArrayList<RegisterType>()
		var a = root.component
		var memoryMaps = a.memoryMaps
		for (m : memoryMaps.eContents.filter(MemoryMapType)) {
			var mmap = m as MemoryMapType
			var listAB = mmap.addressBlock

			for (ab : listAB) {
				registerTypes.addAll(ab.register)
			}
		}
		registerTypes
	}

	def static toText(com.st.stellar.pinmap.pinmapDsl.Include imp) {
		val DocumentRoot importedRegister = ResourceUtils.resourceToEObject(ResourceUtils.openImport(imp.eResource,
			imp.importURI)) as DocumentRoot;
		return '''
			Register model file: «imp.importURI»:
			«IF importedRegister === null»
				File cannot be opened.
			«ELSE»
				Number of registers: «importedRegister.registers.size»
			«ENDIF»
		''';
	}

	val String[] columns = #["#", "PAD", "Pin identifier", "Active", "Function direction", "FunctionName", "Peripheral",
		"Function alternate", "Pin IO advanced settings", "Pin MUX advanced settings"]

	def generateExcel(
		String filename,
		PinSettings settings
	) {
		_pinCfgHelper = PinCfgHelper.instance
		var workbook = new HSSFWorkbook()

		/* CreationHelper helps us create instances of various things like DataFormat, 
		 Hyperlink, RichTextString etc, in a format (HSSF, XSSF) independent way */
		var createHelper = workbook.creationHelper

		// Create a Sheet
		var sheet = workbook.createSheet("PinConfiguration")

		// Create a Font for styling header cells
		val headerFont = workbook.createFont()
		headerFont.setFontHeightInPoints(Short.parseShort("14"));
		headerFont.setColor(IndexedColors.RED.getIndex());

		// Create a CellStyle with the font
		var headerCellStyle = workbook.createCellStyle();
		headerCellStyle.setFont(headerFont);

		// Create a Row
		val headerRow = sheet.createRow(0);

		// Create cells
		for (var i = 0; i < columns.length; i++) {
			val cell = headerRow.createCell(i);
			cell.setCellValue(columns.get(i));
			cell.setCellStyle(headerCellStyle);
		}

		// Create Cell Style for formatting Date
		val dateCellStyle = workbook.createCellStyle();
		dateCellStyle.setDataFormat(createHelper.createDataFormat().getFormat("dd-MM-yyyy"));

		// Create the entries in the sheet
		var rowNb = 1
		for (mod : PinCfgHelper.getAllModules(settings))
			for (ps : mod.pins) {
				val row = sheet.createRow(rowNb++);
				val pin = PinCfgHelper.getPinForConfiguration(ps);
				for (var i = 0; i < columns.length; i++) {
					// String text = provider.getColumnText(pin, i);
					// Label data = new Label(i, row, text, formatData);
					// sheet.addCell(data);
					row.createCell(0).setCellValue(pin.name);
					row.createCell(1).setCellValue(ps.pad.label);
					row.createCell(2).setCellValue(ps.name);
					row.createCell(3).setCellValue("true");
					row.createCell(4).setCellValue(ps.config.direction.getName);
					row.createCell(5).setCellValue(ps.functionName);

					val peripheral = ps.associatedPeripheral
					row.createCell(6).setCellValue(peripheral);
					row.createCell(7).cellValue = ps.functionName
					row.createCell(8).cellValue = ps.IODetails
					row.createCell(9).cellValue = ps.MUXDetails
				}
			}

		val incBoard = PinCfgHelper.getBoard(settings.includedBoard) 
		for (Pin pin : incBoard.package.pins) {
			val cpin = pin.pinConfigurations.get(0);
			if (!(cpin instanceof NotConnectedPin)) {
				val row = sheet.createRow(rowNb++);
				for (var i = 0; i < columns.length; i++) {
					// String text = provider.getColumnText(pin, i);
					// Label data = new Label(i, row, text, formatData);
					// sheet.addCell(data);
					row.createCell(0).setCellValue(pin.name)
					if( cpin instanceof ConfigurablePin) {
						row.createCell(1).setCellValue((cpin as ConfigurablePin).ref.name);
					} else {
						row.createCell(1).setCellValue("");
					}
					row.createCell(3).setCellValue("NONE")
					row.createCell(4).setCellValue(direction(cpin))
					row.createCell(5).setCellValue(valueOfCell(cpin))
				}
			}
		}

// Resize all columns to fit the content size
		for (var i = 0; i < columns.length; i++) {
			sheet.autoSizeColumn(i);
		}

		// Write the output to a file, then read it back to get its CharSequence
		try {
			val fileOut = new FileOutputStream(new File(filename))
			workbook.write(fileOut)
			fileOut.close

		} catch (FileNotFoundException e) {
			println(e.message)
		}
	}

	dispatch def direction(ConfigurablePin config) {
		""
	}

	dispatch def direction(SystemFunctionPin config) {
		config.direction.getName
	}

	dispatch def direction(SpecialPin config) {
		config.direction.getName
	}

	dispatch def direction(NotConnectedPin config) {
		"NotConnectedPin"
	}

	dispatch def valueOfCell(ConfigurablePin config) {
		config.ref.name
	}

	dispatch def valueOfCell(SystemFunctionPin config) {
		config.ref.name
	}

	dispatch def valueOfCell(SpecialPin config) {
		config.ref.name
	}

	dispatch def valueOfCell(NotConnectedPin config) {
		"NotConnectedPin"
	}

	def void closeStream(Closeable stream) {
		if (stream !== null) {
			try {
				stream.close();
			} catch (IOException e) {
				// Ignore
			}
		}
	}

	static def IProject getProject(EObject model) {
		var IProject project = null
		val uri = model.eResource().getURI()
		if (!uri.isFile()) {
			project = null;
		}
		val projectName = uri.segmentsList().get(1).toString();
		project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		return project;
	}

	static def void updateExplorer(EObject model) {

		val project = getProject(model);
		if (project !== null) {
			try {
				project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor())
			} catch (CoreException e) {
				// TODO Auto-generated catch block
				e.printStackTrace()
			}
		}
	}

}
