package com.st.stellar.pinmap.pincfg.glsp.server.model

import com.google.common.collect.HashMultimap
import com.google.common.collect.Multimap
import com.google.inject.Inject
import com.st.stellar.pinmap.generator.GeneratePinmapView
import com.st.stellar.pinmap.helpers.PinCfgHelper
import com.st.stellar.pinmap.helpers.Utils
import com.st.stellar.pinmap.pinCfg.Board
import com.st.stellar.pinmap.pinCfg.PinConfiguration
import com.st.stellar.pinmap.pinCfg.PinSetting
import com.st.stellar.pinmap.pinmapDsl.ConfigurablePin
import com.st.stellar.pinmap.pinmapDsl.Direction
import com.st.stellar.pinmap.pinmapDsl.Package
import com.st.stellar.pinmap.pinmapDsl.Pin
import com.st.stellar.pinmap.pinmapDsl.SpecialPin
import com.st.stellar.pinmap.pinmapDsl.SystemFunctionPin
import com.st.stellar.pinmap.validation.PinCfgValidationMessageAcceptor
import com.st.stellar.pinmap.validation.PinCfgValidator
import java.util.ArrayList
import java.util.List
import org.eclipse.emf.common.util.Diagnostic
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EStructuralFeature
import org.eclipse.emf.ecore.util.Diagnostician
import org.eclipse.glsp.graph.GModelElement

abstract class GeneratePinSettings {

	@Inject extension Utils
	val genPinmapView = new GeneratePinmapView

	public val pinWidthLQFP = 24
	public val pinHeightLQFP = 12
	public val pinMarginWidthLQFP = 7
	public val pinMarginHeightLQFP = 3
	public val fontSizeLQFP = pinWidthLQFP / 2
	public val generalOffsetXLQFP = -25
	public val generalOffsetYLQFP = -15

	@Inject extension PinCfgHelper;

	static class PinSettingsErrorManager extends PinCfgValidationMessageAcceptor {
		public val Multimap<EObject, String> errors
		public val Multimap<EObject, String> infos
		public val Multimap<EObject, String> warnings

		new() {
			super(null)
			errors = HashMultimap.create
			infos = HashMultimap.create
			warnings = HashMultimap.create
			acceptsExternalResource = true
		}

		override collectError(String message, EObject object, EStructuralFeature feature, int index, String code,
			String... issueData) {
			errors.put(object, message)
		}

		override collectError(String message, EObject object, int offset, int length, String code,
			String... issueData) {
			errors.put(object, message)
		}

		override collectInfo(String message, EObject object, EStructuralFeature feature, int index, String code,
			String... issueData) {
			infos.put(object, message)
		}

		override collectInfo(String message, EObject object, int offset, int length, String code, String... issueData) {
			infos.put(object, message)
		}

		override collectWarning(String message, EObject object, EStructuralFeature feature, int index, String code,
			String... issueData) {
			warnings.put(object, message)
		}

		override collectWarning(String message, EObject object, int offset, int length, String code,
			String... issueData) {
			warnings.put(object, message)
		}

	}

//	def dispatch String getRefId(PinSettings settings) {
//		var n = (PinCfgHelper.getBoard(settings.includedBoard).package.eContainer as Family).name
//		n = n.replaceAll('x', PinCfgHelper.getBoard(settings.includedBoard).package.name)
//		var a = n.indexOf('_')
//		n = n.substring(0, a)
//		n
//	}
//
//	def dispatch String getRefId(PinSetting setting) {
//		setting.pin.pinConfigurations.get(0).getRefId
//	}
//
//	def dispatch String getRefId(ConfigurablePin pin) {
//		var cfg = pin.ref as Configuration
//		cfg.name
//
//	}

	new() {
		errors = new StringBuilder("errors:\n")
		warnings = new StringBuilder("warnings:\n")
	}

	protected var StringBuilder errors

	def error(String message) {
		errors.append("error: " + message + "\n")
	}

	protected var StringBuilder warnings

	def warning(String message) {
		warnings.append("warning: " + message + "\n")
	}

	def generateReport() {
		return errors.append(warnings)
	}

	def boolean generatePins(GModelElement newRoot, PinCfgModelState modelState, PinConfiguration pinConfiguration) {return false}

	def isBoardPin(PinSetting pin) {
		_pinCfgHelper = PinCfgHelper.instance
		if (pin.eContainer.eContainer instanceof Board) {
			return true
		} else {
			return false
		}
	}

	def boolean addConfiguredPin(Package pack, GModelElement newRoot, PinCfgModelState modelState, PinSetting pin,
		PinSettingsErrorManager errorsMap) {return false}

	def GModelElement generatePackage(GModelElement newRoot, PinCfgModelState modelState, Package p, List<PinSetting> configuredPins,
		String additional) {null}

	def String getTooltip(Pin pin) {
		var tooltip = ""
		for (cfg : pin.pinConfigurations) {
			if (cfg instanceof SystemFunctionPin) {
				val sfp = cfg as SystemFunctionPin
				tooltip += sfp.ref.label + " (" + sfp.direction.getName + ")"
				tooltip += "<br/>"
			} else if (cfg instanceof SpecialPin) {
				val sp = cfg as SpecialPin
				tooltip += sp.ref.label + " (" + sp.direction.getName + ")"
				tooltip += "<br/>"
			} else if (cfg instanceof ConfigurablePin) {
				val sfp = cfg as ConfigurablePin
				tooltip += "<b>" + pin.name + ": " + sfp.ref.label + "</b><br/><pre>"
				for (c : sfp.ref.functions) {
					tooltip += "   " + c.name.name + "<br/>"
				}
				tooltip += "</pre>"
			} else {
				tooltip += "   " + cfg.toString + "<br/>"
			}
		}
		''' onmousemove="showTooltip(evt, '«tooltip»');" onmouseout="hideTooltip();"'''
	}

	def boolean relatedToPinSetting(Diagnostic diag, PinSetting ps) {
		if(ps === null) return false
		if(ps.config === null) return false
		if(ps.ioFields === null) return false

		var res = false
		val List<EObject> objList = newArrayList
		objList.add(ps)
		objList.add(ps.config)
		objList.add(ps.ioFields)
		for (reg : ps.ioFields.reg) {
			objList.add(reg)
		}
		if (objList.contains(diag.data.get(0))) {
			res = true
		}
		return res
	}

	def PinSettingsErrorManager generateErrors(GModelElement newRoot, PinCfgModelState modelState,
		List<PinSetting> configuredPins) {

		val pinConfiguration = modelState.pinConfiguration
		val errorManager = new PinSettingsErrorManager()
		PinCfgValidator.checkNoDuplicatedPinNames(pinConfiguration, errorManager)
		PinCfgValidator.checkUniqueFunctionUsage(pinConfiguration, errorManager)
		PinCfgValidator.checkPinAlreadyUsed(pinConfiguration, errorManager)

		for (ps : configuredPins) {
			val diags = new ArrayList<Diagnostic>();

			PinCfgValidator.checkMissingRegInstances(ps, errorManager)

			if (ps.isBoardPin) {
				val errorCfg = Diagnostician.INSTANCE.validate(pinConfiguration);
				val errorsCfg = errorCfg.getChildren();
				diags.addAll(errorsCfg.filter[it|it.relatedToPinSetting(ps)]);
				for (Diagnostic e : errorsCfg) {
					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
				}
			} else {
				val errorPs = Diagnostician.INSTANCE.validate(ps);
				val errorsPs = errorPs.getChildren();
				diags.addAll(errorsPs.filter[it|it.relatedToPinSetting(ps)]);
				for (Diagnostic e : errorsPs) {
					diags.addAll(e.getChildren().filter[it|it.relatedToPinSetting(ps)]);
				}
			}
			if (diags.size > 0) {
				diags.filter[d|d.severity == Diagnostic.ERROR].map(it|it.message).forEach(
					it |
						errorManager.errors.put(ps, it)
				)
				diags.filter[d|d.severity == Diagnostic.WARNING].map(it|it.message).forEach(
					it |
						errorManager.warnings.put(ps, it)
				)
				diags.filter[d|d.severity == Diagnostic.INFO].map(it|it.message).forEach(
					it |
						errorManager.infos.put(ps, it)
				)
			}
		}
		return errorManager
	}

	static protected val directionNames = newLinkedHashMap(Direction.INPUT -> "in", Direction.OUTPUT -> "out",
		Direction.INPUT_OUTPUT -> "inout", Direction.INPUT_OR_OUTPUT -> "inout", Direction.GROUND -> "ground",
		Direction.POWER -> "power", Direction.INPUT_ALTERNATE -> "", Direction.OUTPUT_ALTERNATE -> "")

}
