package com.st.stellar.pinmap.pincfg.glsp.server.model

import com.google.inject.Inject
import com.st.stellar.pinmap.generator.GeneratePinmapView
import com.st.stellar.pinmap.helpers.PinCfgHelper
import com.st.stellar.pinmap.helpers.Utils
import com.st.stellar.pinmap.pinCfg.PinConfiguration
import com.st.stellar.pinmap.pinCfg.PinSetting
import com.st.stellar.pinmap.pincfg.glsp.server.PinCfgModelTypes
import com.st.stellar.pinmap.pinmapDsl.ConfigurablePin
import com.st.stellar.pinmap.pinmapDsl.Family
import com.st.stellar.pinmap.pinmapDsl.NotConnectedPin
import com.st.stellar.pinmap.pinmapDsl.Package
import com.st.stellar.pinmap.pinmapDsl.Pin
import com.st.stellar.pinmap.pinmapDsl.PinConfig
import com.st.stellar.pinmap.pinmapDsl.SpecialPin
import com.st.stellar.pinmap.pinmapDsl.SystemFunctionPin
import java.util.ArrayList
import java.util.Arrays
import java.util.List
import java.util.Map
import org.eclipse.glsp.graph.DefaultTypes
import org.eclipse.glsp.graph.GModelElement
import org.eclipse.glsp.graph.builder.impl.GLabelBuilder
import org.eclipse.glsp.graph.builder.impl.GLayoutOptions
import org.eclipse.glsp.graph.builder.impl.GNodeBuilder
import org.eclipse.glsp.graph.util.GConstants

class GeneratePinSettingsBGA extends GeneratePinSettings {

	@Inject extension Utils
	val genPinmapView = new GeneratePinmapView

	public val pinWidthBGA = 40
	public val pinHeightBGA = 40
	public val pinMarginWidthBGA = 3
	public val pinMarginHeightBGA = 3
	public val fontSizeBGA = pinWidthBGA / 3

	@Inject extension PinCfgHelper;

	val rowLetters = "A,B,C,D,E,F,G,H,J,K,L,M,N,P,R,T,U,V,W,Y,AA,AB,AC,AD,AE,AF,AG,AH,AJ,AK"

	val rowLettersListOrdered = Arrays.asList(rowLetters.split(","));

	// flip 
	val rowLettersList = rowLettersListOrdered.reverseView

	public List<String> rowEnabledLetters = new ArrayList<String>()

	int deltaY

	override boolean generatePins(GModelElement newRoot, PinCfgModelState modelState,
		PinConfiguration pinConfiguration) {
		_pinCfgHelper = PinCfgHelper.instance
		_utils = Utils.instance
		val modules = PinCfgHelper.getAllModules(pinConfiguration.pinSetting)

		var pack = pinConfiguration.pack

		if (modules.size > 0) {
			val configuredPins = modules.flatMap[it|it.pins].toList
			val packageNode = generatePackage(newRoot, modelState, pack, configuredPins, "")

			val errorsMap = generateErrors(packageNode, modelState, configuredPins)
			var status = true
			for (pin : configuredPins) {
				status = status && addConfiguredPin(pack, packageNode, modelState, pin, errorsMap)
			}
			return status
		} else {
			return generatePackage(newRoot, modelState, pinConfiguration.pinSetting.pack, newArrayList, "") !== null
		}
	}

	override boolean addConfiguredPin(Package pack, GModelElement newRoot, PinCfgModelState modelState, PinSetting ps,
		PinSettingsErrorManager errorsMap) {

		try {
			var pin = PinCfgHelper.getPinForConfiguration(ps);

			var x = genPinmapView.getX(pin) * (pinWidthBGA + pinMarginWidthBGA) - pinWidthBGA / 2
			var y = (genPinmapView.getY(pin) - deltaY) * (pinHeightBGA + pinMarginHeightBGA) + 3 * pinHeightBGA / 4

			val id = modelState.getIndex().indexEObject(ps.pad);

			// println('''Configured pin: «pin.name»: «id»''')
			val config = ps.pad

			var cssClasses = newArrayList
			val nodeClasses = newArrayList

			if (ps.isBoardPin) {
				cssClasses.add("board")
			} else {
				cssClasses.add("cPin")
			}

			var PinNodeBuilder pinNode = null
			val padName = config.getCfgName(4)

			val errors = errorsMap.errors.get(ps)
			val warnings = errorsMap.warnings.get(ps)
			val infos = errorsMap.infos.get(ps)

			// Search for duplicated GModelElement
			val graphObject = newRoot.children.findFirst[it.id.equals(id)]
			if (graphObject === null) {
				if (errors.size > 0 || warnings.size > 0 || infos.size > 0) {
					pinNode = new PinNodeBuilder(PinCfgModelTypes.PIN_ERROR)
					pinNode.addArgument("name", ps.pad.name)
					if (errors.size > 0) {
						pinNode.addArgument("error", errors.join(","))
						nodeClasses.add("error");
						cssClasses.add("error");
					} else if (warnings.size > 0) {
						pinNode.addArgument("warning", warnings.join(","))
						nodeClasses.add("warning");
						cssClasses.add("warning");
					} else if (infos.size > 0) {
						pinNode.addArgument("info", infos.join(","))
						nodeClasses.add("info");
						cssClasses.add("info");
					}
					newRoot.getChildren().add(
						pinNode.id(id).addCssClasses(cssClasses).add(
							new GLabelBuilder(PinCfgModelTypes.PIN_NAME).addCssClass("configured-text").text(padName).
								id(id + "_pin_label").position((4 - padName.length) * 2 + 6,
									pinHeightBGA / 4 + pinMarginHeightBGA).build()).size(pinWidthBGA, pinHeightBGA).
							position(x, y).build());
				} else {
					pinNode = new PinNodeBuilder(PinCfgModelTypes.PIN).id(id)
					pinNode.addArgument("name", ps.pad.name)
					newRoot.getChildren().add(pinNode //
					.addCssClasses(cssClasses) //
					.add(
						new GLabelBuilder(PinCfgModelTypes.PIN_NAME).addCssClass("configured-text").text(padName).id(
							id + "_pin_label").position((4 - padName.length) * 2 + 6,
							pinHeightBGA / 4 + pinMarginHeightBGA).build()).size(pinWidthBGA, pinHeightBGA).
						position(x, y).build());
				}
			} else {
				println("coucou")
			}
			return true
		} catch (Exception e) {
			return false
		}
	}

	def void generateLegend(GModelElement newRoot, PinCfgModelState modelState, double width) {
		_pinCfgHelper = PinCfgHelper.instance
		val node = new PinNodeBuilder(PinCfgModelTypes.LEGEND_TYPE).position(width - 150, -60).id("legend")
		node.addArgument("width", width)
		val state = getCurrentLegendState(modelState)
		node.addArgument("state", state)

		newRoot.children.add(node.build())
	}

	def getCurrentLegendState(PinCfgModelState modelState) {
		val key = modelState.getPinConfiguration().eResource().getURI().path()
		PreferenceManager.getPreference(key, "false")
	}

	override GModelElement generatePackage(GModelElement newRoot, PinCfgModelState modelState, Package p,
		List<PinSetting> configuredPins, String additional) {

		try {
			var GModelElement res = null

			var id = modelState.getIndex().indexEObject(p);

			// Add axes
			var nbColumns = 0
			for (pin : p.pins) {
				val pName = pin.name
				val letter = pName.replaceAll("[0-9]", "")
				nbColumns = Math.max(nbColumns,rowLettersListOrdered.indexOf(letter)+1)
			}
			nbColumns++
			
			for(var i = 1;i<nbColumns; i++) {
				val letter = rowLettersListOrdered.get(i-1)
				rowEnabledLetters.add(letter)
			}
			deltaY = rowLettersList.length - rowEnabledLetters.length

			val nbRows = nbColumns
			val width = (pinWidthBGA + pinMarginWidthBGA) * nbColumns
			val height = (pinHeightBGA + pinMarginHeightBGA) * nbRows

			val cssClasses = newArrayList
			cssClasses.add("gpiocfg-package")
			val packageName = new GLabelBuilder(PinCfgModelTypes.PACKAGE_NAME).text((p.eContainer as Family).label +
				p.name).id("packageId").addCssClasses(cssClasses).position(width / 2, -2 * pinHeightBGA)
			newRoot.children.add(packageName.build())

			val packageNodeBuilder = new GNodeBuilder(PinCfgModelTypes.PACKAGE).id(id).addCssClass("pincfg-package").
				size(width, height).position(0, 0)

			val packageNode = packageNodeBuilder.build()

			generateLegend(newRoot, modelState, width);

			res = packageNode
			newRoot.children.add(res)

			for (pin : p.pins) {
				val config = pin.pinConfigurations.get(0) as PinConfig
				if (!(config instanceof NotConnectedPin)) {
					generatePin(res, modelState, configuredPins, pin)
				}
			}

			for (var i = 1; i < nbColumns; i++) {
				newRoot.getChildren().add(new GNodeBuilder(PinCfgModelTypes.AXIS_INDEX) //
				.add(new GLabelBuilder(DefaultTypes.LABEL).text(String.valueOf(i)).build()).addCssClass(
					"axis-index-north") //
				.position(i * (pinWidthBGA + pinMarginWidthBGA) - 5, -12).build());

				newRoot.getChildren().add(new GNodeBuilder(PinCfgModelTypes.AXIS_INDEX) //
				.addCssClass("axis-index-south") //
				.add(new GLabelBuilder(DefaultTypes.LABEL).text(String.valueOf(i)).build()).layout(
					GConstants.Layout.VBOX, Map.of(GLayoutOptions.KEY_PADDING_RIGHT, 5)) //
				.position(i * (pinWidthBGA + pinMarginWidthBGA) - 5,
					((pinHeightBGA + pinMarginHeightBGA) * (nbRows + 1)) - pinHeightBGA + 8).build());
			}

			for (var i = rowEnabledLetters.size as int; i > 0; i--) {
				val indexName = rowLettersList.get(i + deltaY - 1)
				newRoot.getChildren().add(new GNodeBuilder(PinCfgModelTypes.AXIS_INDEX) //
				.addCssClass("axis-index-west") //
				.add(
					new GLabelBuilder(DefaultTypes.LABEL).text(indexName).position(
						30 - (indexName.length * 5), 0).build()
				).layout(GConstants.Layout.HBOX).layoutOptions(new GLayoutOptions().vAlign(GConstants.VAlign.CENTER)).
					position(-pinWidthBGA + pinMarginWidthBGA, i * (pinHeightBGA + pinMarginHeightBGA)).build());

				newRoot.getChildren().add(new GNodeBuilder(PinCfgModelTypes.AXIS_INDEX).addCssClass("axis-index-east") //
				.add(
					new GLabelBuilder(DefaultTypes.LABEL).text(indexName).build()
				).layout(GConstants.Layout.HBOX, Map.of(GLayoutOptions.KEY_H_ALIGN, 0)) //
				.position((pinHeightBGA + pinMarginHeightBGA) * (nbRows + 1) - pinWidthBGA + pinMarginWidthBGA,
					i * (pinHeightBGA + pinMarginHeightBGA)).build());
			}
			res
		} catch (Exception e) {
			return null
		}
	}

	def getX(Pin pin) {
		val id = pin.name
		val number = id.replaceAll("[a-zA-Z]", "")
		Integer.parseInt(number)
	}

	def getY(Pin pin) {
		val id = pin.name
		val letter = id.replaceAll("[0-9]", "")
		rowLettersList.indexOf(letter) - deltaY
	}

	def generatePin(GModelElement newRoot, PinCfgModelState modelState, List<PinSetting> configuredPins, Pin pin) {

		var String type = null
		var PinSetting foundPin = null
		val config = pin.pinConfigurations.get(0) as PinConfig
		var String idd = modelState.getIndex().indexEObject(pin);
		var String name = pin.name
		if (SpecialPin.isInstance(config)) {
			type = "special-pin"
		} else if (SystemFunctionPin.isInstance(config)) {
			type = "systemfunction-pin"
		} else if (ConfigurablePin.isInstance(config)) {
			val cfg = ConfigurablePin.cast(config)
			idd = modelState.getIndex().indexEObject(cfg.ref)
			name = cfg.ref.name
			foundPin = configuredPins.findFirst[it|it.pad.name.equals(cfg.ref.name)]
			if (foundPin === null) {
				type = "configurable-pin"
			}
		}
		if (type !== null) {
			val padName = config.getCfgName(4)

			var x = genPinmapView.getX(pin) * (pinWidthBGA + pinMarginWidthBGA) - pinWidthBGA / 2
			var y = (genPinmapView.getY(pin) - deltaY) * (pinHeightBGA + pinMarginHeightBGA) + 3 * pinHeightBGA / 4

			val pinNode = new GNodeBuilder(PinCfgModelTypes.PIN)
			pinNode.addArgument("name", name)
			newRoot.getChildren().add(pinNode //
			.id(idd) //
			.addCssClass(type) //
			.add(
				new GLabelBuilder(DefaultTypes.LABEL).text(padName).position((4 - padName.length) * 2 + 6,
					pinHeightBGA / 4 + pinMarginHeightBGA).id(idd + "_pin_label").build()
			).size(pinWidthBGA, pinHeightBGA).position(x, y).build());
		}
	}

}
