package com.st.stellar.pinmap.pincfg.glsp.server.model

import com.google.inject.Inject
import com.st.stellar.pinmap.generator.GeneratePinmapViewLQFP
import com.st.stellar.pinmap.helpers.PinCfgHelper
import com.st.stellar.pinmap.helpers.Utils
import com.st.stellar.pinmap.pinCfg.PinConfiguration
import com.st.stellar.pinmap.pinCfg.PinSetting
import com.st.stellar.pinmap.pincfg.glsp.server.PinCfgModelTypes
import com.st.stellar.pinmap.pinmapDsl.ConfigurablePin
import com.st.stellar.pinmap.pinmapDsl.Family
import com.st.stellar.pinmap.pinmapDsl.LQFPPackage
import com.st.stellar.pinmap.pinmapDsl.NotConnectedPin
import com.st.stellar.pinmap.pinmapDsl.Package
import com.st.stellar.pinmap.pinmapDsl.Pin
import com.st.stellar.pinmap.pinmapDsl.PinConfig
import com.st.stellar.pinmap.pinmapDsl.SpecialPin
import com.st.stellar.pinmap.pinmapDsl.SystemFunctionPin
import java.util.List
import java.util.regex.Pattern
import org.eclipse.glsp.graph.GModelElement
import org.eclipse.glsp.graph.builder.impl.GLabelBuilder
import org.eclipse.glsp.graph.builder.impl.GNodeBuilder
import com.st.stellar.pinmap.pinCfg.PinCfgPackage

class GeneratePinSettingsLQFP extends GeneratePinSettings {

	@Inject extension Utils
	val genPinmapView = new GeneratePinmapViewLQFP

	public val pinWidthLQFP = 24
	public val pinHeightLQFP = 12
	public val pinMarginWidthLQFP = 7
	public val pinMarginHeightLQFP = 3
	public val fontSizeLQFP = pinWidthLQFP / 2
	public val generalOffsetXLQFP = -25
	public val generalOffsetYLQFP = -15

	@Inject extension PinCfgHelper;

	override boolean generatePins(GModelElement newRoot, PinCfgModelState modelState,
		PinConfiguration pinConfiguration) {
		_pinCfgHelper = PinCfgHelper.instance
		val modules = PinCfgHelper.getAllModules(pinConfiguration.pinSetting)
		var List<PinSetting> configuredPins = newArrayList
		if (modules.size > 0) {
			configuredPins = modules.flatMap[it|it.pins].toList
		}

		val pack = pinConfiguration.pinSetting.pack
		return generatePackage(newRoot, modelState, pack, configuredPins, "") !== null
	}

	override generatePackage(GModelElement newRoot, PinCfgModelState modelState, Package p,
		List<PinSetting> configuredPins, String additional) {

		try {
			var GModelElement res = null
			var id = modelState.getIndex().indexEObject(p);

			val nbColumns = p.pins.size / 4
			val nbRows = nbColumns
			val pinSize = Math.min(pinWidthLQFP, pinHeightLQFP);
			val width = (pinSize + 2 * pinMarginHeightLQFP) * nbRows + pinMarginHeightLQFP
			val height = (pinSize + 2 * pinMarginHeightLQFP) * nbRows + pinMarginHeightLQFP

			val cssClasses = newArrayList
			cssClasses.add("gpiocfg-package")

			val legendNode = generateLegend(newRoot, modelState, width);

			val packageNodeBuilder = new GNodeBuilder(PinCfgModelTypes.LQFP_PACKAGE).size(width, width).add(legendNode).
				add(
					new GLabelBuilder(PinCfgModelTypes.LQFP_PACKAGE_NAME).text((p.eContainer as Family).name + p.name).
						id("packageId").addCssClasses(cssClasses).position(width / 2, height / 2).build()).id(id)

			val packageNode = packageNodeBuilder.build()

			res = packageNode
			newRoot.children.add(packageNode)

			val errorsMap = generateErrors(newRoot, modelState, configuredPins)
			for (pin : p.pins) {
				val config = pin.pinConfigurations.get(0) as PinConfig
				if (!(config instanceof NotConnectedPin)) {
					generatePin(newRoot, modelState, configuredPins, pin, errorsMap)
				}
			}

			generateLegend(newRoot, modelState, width);
			res
		} catch (Exception e) {
			return null
		}
	}

	def GModelElement generateLegend(GModelElement parent, PinCfgModelState modelState, int width) {
		_pinCfgHelper = PinCfgHelper.instance
		val node = new PinNodeBuilder(PinCfgModelTypes.LQFP_LEGEND_TYPE).position(width + pinWidthLQFP / 2, -50)
		node.addArgument("width", width)

		val res = node.build()
		parent.children.add(res)

		res
	}

	def generatePin(GModelElement newRoot, PinCfgModelState modelState, List<PinSetting> configuredPins, Pin pin,
		PinSettingsErrorManager errorsMap) {

		val ps = configuredPins.findFirst [ it |
			{
				val p = PinCfgHelper.getPinForConfiguration(it)
				p.name !== null && p.name.equals(pin.name)
			}
		]
		if (ps === null) {
			generateNotConfiguredPin(newRoot, modelState, pin, errorsMap)
		} else {
			val pack = pin.eContainer as Package
			generateConfiguredPin(pack, newRoot, modelState, ps, errorsMap)
		}
	}

	def generateNotConfiguredPin(GModelElement newRoot, PinCfgModelState modelState, Pin pin,
		PinSettingsErrorManager errorsMap) {

		val cssClasses = newArrayList
		val nodeClasses = newArrayList
		cssClasses.add("pinNone")
		nodeClasses.add("pinNone")

		val config = pin.pinConfigurations.get(0) as PinConfig
		var x = 0
		var y = 0
		val pattern = Pattern.compile("^.*_([0-9]+)$")
		var matcher = pattern.matcher(pin.name)
		if (matcher.find()) {
			val p = pin.eContainer as LQFPPackage
			val size = p.pins.size / 4

			var number = matcher.group(1)
			val index = Integer.parseInt(number)

			var side = ""
			if (index <= size) { // west
				side = "west"
			} else if (index <= size * 2 && index > size) { // south
				side = "south"
			} else if (index <= size * 3 && index > size * 2) { // east
				side = "east"
			} else if (index > size * 3) { // north
				side = "north"
			}

			var PinNodeBuilder pinNode = null

			var GLabelBuilder gName = null
			var GLabelBuilder gLabel = null
			var GLabelBuilder gDirection = null
			gLabel = null
			if (SpecialPin.isInstance(config)) {
				val id = modelState.getIndex().indexEObject(pin);
				pinNode = new PinNodeBuilder(PinCfgModelTypes.LQFP_PIN).id(id)
				pinNode.addArgument("side", side)
				pinNode.addArgument("name", pin.name)

				val sp = SpecialPin.cast(config)
				val directionName = GeneratePinSettings.directionNames.get(sp.direction) + "_" + side

				nodeClasses.add("nonConfigurablePin")
				gName = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_NAME).text(sp.ref.name).id(id + "_pin_name")
				gName.addArgument("side", side)
				gLabel = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_ID).text(pin.pinNameLQFP).id(id + "_pin_label")
				gLabel.addArgument("side", side)
				gDirection = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_DIRECTION).id(id + "_pin_direction")
				gDirection.addArgument("direction", directionName)
			} else if (SystemFunctionPin.isInstance(config)) {
				val id = modelState.getIndex().indexEObject(pin);
				pinNode = new PinNodeBuilder(PinCfgModelTypes.LQFP_PIN).id(id)
				pinNode.addArgument("side", side)
				pinNode.addArgument("name", pin.name)

				val sp = SystemFunctionPin.cast(config)

				val directionName = GeneratePinSettings.directionNames.get(sp.direction) + "_" + side
				nodeClasses.add("nonConfigurablePin")

				gName = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_NAME).text(sp.ref.name).id(id + "_pin_name")
				gName.addArgument("side", side)

				gLabel = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_ID).text(pin.pinNameLQFP).id(id + "_pin_label")
				gLabel.addArgument("side", side)

				gDirection = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_DIRECTION).id(id + "_pin_direction")
				gDirection.addArgument("direction", directionName)
			} else if (NotConnectedPin.isInstance(config)) {
				val id = modelState.getIndex().indexEObject(pin);
				pinNode = new PinNodeBuilder(PinCfgModelTypes.LQFP_PIN).id(id)
				pinNode.addArgument("side", side)
				pinNode.addArgument("name", pin.name)

				gName = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_NAME).text("not connected").id(id + "_pin_name")
				gName.addArgument("side", side)

			} else if (ConfigurablePin.isInstance(config)) {
				val cp = ConfigurablePin.cast(config)
				val id = modelState.getIndex().indexEObject(cp.ref);
				pinNode = new PinNodeBuilder(PinCfgModelTypes.LQFP_PIN).id(id)
				pinNode.addArgument("side", side)
				pinNode.addArgument("name", cp.ref.name)
				gLabel = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_ID).text(pin.pinNameLQFP).id(id + "_pin_label")
				gLabel.addArgument("side", side)
			}

			var imageOffsetX = 0
			var imageOffsetY = 0
			// println(index)
			if (index <= size) { // west
				x = 0
				y = index
				x *= (pinWidthLQFP + 2 * pinMarginWidthLQFP)
				x += -3
				y *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
				imageOffsetY = 1
				imageOffsetX = - pinWidthLQFP / 2
				side = "west"

				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index <= size * 2 && index > size) { // south
				y = (size + 1) * (pinHeightLQFP + 2 * pinMarginHeightLQFP) + pinMarginHeightLQFP
				y += 1
				x = index - size
				x *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
				x += pinHeightLQFP
				// y *= (pinHeight + 2 * pinMarginWidth)
				// y += pinHeight - pinMarginHeight
				imageOffsetX = 0
				imageOffsetY += pinWidthLQFP + 2
				side = "south"
				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).position(pinHeightLQFP + pinMarginHeightLQFP, 0).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index <= size * 3 && index > size * 2) { // east
				x = size + 2
				y = size * 3 - index + 1
				x *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
				x -= 4
				y *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
				imageOffsetX = pinWidthLQFP + 2
				imageOffsetY = 1
				side = "east"

				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).position(pinWidthLQFP + pinMarginWidthLQFP, 0).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			} else if (index > size * 3) { // north
				x = (size * 4) - index + 1
				y = 1

				x *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
				x += pinHeightLQFP

				y *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
				y -= 2 * (pinHeightLQFP + pinMarginHeightLQFP)
				y -= 1
				imageOffsetX = 0
				imageOffsetY = -pinHeightLQFP - 2 // (pinWidth + pinHeight + pinMarginWidth)
				side = "north"
				if (gLabel !== null) {
					pinNode.add(gLabel.addCssClasses(cssClasses).build)
				}
				if (gName !== null) {
					pinNode.add(gName.addCssClasses(cssClasses).build)
				}
				if (gDirection !== null) {
					pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
				}
			}

			nodeClasses.add(side)
			x += generalOffsetXLQFP
			y += generalOffsetYLQFP

			newRoot.getChildren().add(pinNode.addCssClasses(nodeClasses).position(x, y).build)
		}
	}

	def generateConfiguredPin(Package pack, GModelElement newRoot, PinCfgModelState modelState, PinSetting ps,
		PinSettingsErrorManager errorsMap) {

		val pin = Utils.getPinForConfiguration(pack, ps.pad)

		val cssClasses = newArrayList
		val nodeClasses = newArrayList
		if (ps.isBoardPin) {
			cssClasses.add("board")
			nodeClasses.add("board")
		} else {
			cssClasses.add("cPin")
			nodeClasses.add("cPin")
		}

		var x = 0
		var y = 0
		val pattern = Pattern.compile("^.*_([0-9]+)$")

		val id = modelState.getIndex().indexEObject(ps.pad); // pin.name
		// Search for duplicated GModelElement
		val graphObject = newRoot.children.findFirst[it.id.equals(id)]
		if (graphObject === null) {

			var matcher = pattern.matcher(pin.name)
			if (matcher.find()) {
				val p = pin.eContainer as LQFPPackage
				val size = p.pins.size / 4

				var number = matcher.group(1)
				val index = Integer.parseInt(number)

				var side = ""
				if (index <= size) { // west
					side = "west"
				} else if (index <= size * 2 && index > size) { // south
					side = "south"
				} else if (index <= size * 3 && index > size * 2) { // east
					side = "east"
				} else if (index > size * 3) { // north
					side = "north"
				}

				var PinNodeBuilder pinNode = null

				var GLabelBuilder gName = null
				var GLabelBuilder gLabel = null
				var GLabelBuilder gDirection = null
				val directionName = ps.config.direction.literal + "_" + side

				gName = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_NAME).text(ps.name).id(id + "_pin_name")
				gName.addArgument("side", side)

				gLabel = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_ID).text(pin.pinNameLQFP).id(id + "_pin_label")
				gLabel.addArgument("side", side)

				gDirection = new GLabelBuilder(PinCfgModelTypes.LQFP_PIN_DIRECTION).id(id + "_pin_direction")
				gDirection.addArgument("direction", directionName)

				val errors = errorsMap.errors.get(pin)
				val warnings = errorsMap.warnings.get(pin)
				val infos = errorsMap.infos.get(pin)
				if (errors.size > 0 || warnings.size > 0 || infos.size > 0) {
					pinNode = new PinNodeBuilder(PinCfgModelTypes.LQFP_PIN_ERROR).id(id)
					pinNode.addArgument("name", ps.pad.name)
					if (errors.size > 0) {
						pinNode.addArgument("error", errors.join(","))
						nodeClasses.add("error");
						cssClasses.add("error");
					} else if (warnings.size > 0) {
						pinNode.addArgument("warning", warnings.join(","))
						nodeClasses.add("warning");
						cssClasses.add("warning");
					} else if (infos.size > 0) {
						pinNode.addArgument("info", infos.join(","))
						nodeClasses.add("info");
						cssClasses.add("info");
					}
					pinNode.addArgument("side", side)
				} else {
					pinNode = new PinNodeBuilder(PinCfgModelTypes.LQFP_PIN).id(id)
					pinNode.addArgument("side", side)
					pinNode.addArgument("name", ps.pad.name)
				}

				var imageOffsetX = 0
				var imageOffsetY = 0
				// println(index)
				if (index <= size) { // west
					x = 0
					y = index
					x *= (pinWidthLQFP + 2 * pinMarginWidthLQFP)
					x += -3
					y *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
					imageOffsetY = 1
					imageOffsetX = - pinWidthLQFP / 2
					side = "west"

					if (gLabel !== null) {
						pinNode.add(gLabel.addCssClasses(cssClasses).build)
					}
					if (gName !== null) {
						pinNode.add(gName.addCssClasses(cssClasses).build)
					}
					if (gDirection !== null) {
						pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
					}
				} else if (index <= size * 2 && index > size) { // south
					y = (size + 1) * (pinHeightLQFP + 2 * pinMarginHeightLQFP) + pinMarginHeightLQFP
					y += 1
					x = index - size
					x *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
					x += pinHeightLQFP
					// y *= (pinHeight + 2 * pinMarginWidth)
					// y += pinHeight - pinMarginHeight
					imageOffsetX = 0
					imageOffsetY += pinWidthLQFP + 2
					side = "south"
					if (gLabel !== null) {
						pinNode.add(gLabel.addCssClasses(cssClasses).build)
					}
					if (gName !== null) {
						pinNode.add(
							gName.addCssClasses(cssClasses).position(pinHeightLQFP + pinMarginHeightLQFP, 0).build)
					}
					if (gDirection !== null) {
						pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
					}
				} else if (index <= size * 3 && index > size * 2) { // east
					x = size + 2
					y = size * 3 - index + 1
					x *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
					x -= 4
					y *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
					imageOffsetX = pinWidthLQFP + 2
					imageOffsetY = 1
					side = "east"

					if (gLabel !== null) {
						pinNode.add(gLabel.addCssClasses(cssClasses).build)
					}
					if (gName !== null) {
						pinNode.add(
							gName.addCssClasses(cssClasses).position(pinWidthLQFP + pinMarginWidthLQFP, 0).build)
					}
					if (gDirection !== null) {
						pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
					}
				} else if (index > size * 3) { // north
					x = (size * 4) - index + 1
					y = 1

					x *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
					x += pinHeightLQFP

					y *= (pinHeightLQFP + 2 * pinMarginHeightLQFP)
					y -= 2 * (pinHeightLQFP + pinMarginHeightLQFP)
					y -= 1
					imageOffsetX = 0
					imageOffsetY = -pinHeightLQFP - 2 // (pinWidth + pinHeight + pinMarginWidth)
					side = "north"
					if (gLabel !== null) {
						pinNode.add(gLabel.addCssClasses(cssClasses).build)
					}
					if (gName !== null) {
						pinNode.add(gName.addCssClasses(cssClasses).build)
					}
					if (gDirection !== null) {
						pinNode.add(gDirection.position(imageOffsetX, imageOffsetY).build)
					}
				}

				nodeClasses.add(side)
				x += generalOffsetXLQFP
				y += generalOffsetYLQFP

				newRoot.getChildren().add(pinNode.addCssClasses(nodeClasses).position(x, y).build)
			}
		}
	}

	def getPinNameLQFP(Pin pin) {
		val pattern = Pattern.compile("^(.*)_([0-9]+)$")
		val id = pin.name
		var matcher = pattern.matcher(id)
		if (matcher.find()) {
			var number = matcher.group(2)
			return number
		} else {
			return ""
		}
	}
}
